/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at
 *
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.geom.prep

import org.locationtech.jts.algorithm.PointLocator
import org.locationtech.jts.geom.Coordinate
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.util.ComponentCoordinateExtracter
import org.locationtech.jts.noding.SegmentString
import org.locationtech.jts.noding.SegmentStringUtil.extractSegmentStrings

/**
 * Computes the <tt>intersects</tt> spatial relationship predicate
 * for a target [PreparedLineString] relative to other [Geometry] classes.
 * Uses short-circuit tests and indexing to improve performance.
 *
 * @author Martin Davis
 */
internal class PreparedLineStringIntersects(prepLine: PreparedLineString) {
    protected var prepLine: PreparedLineString

    /**
     * Creates an instance of this operation.
     *
     * @param prepPoly the target PreparedLineString
     */
    init {
        this.prepLine = prepLine
    }

    /**
     * Tests whether this geometry intersects a given geometry.
     *
     * @param geom the test geometry
     * @return true if the test geometry intersects
     */
    fun intersects(geom: Geometry): Boolean {
        /**
         * If any segments intersect, obviously intersects = true
         */
        val lineSegStr: MutableList<SegmentString> = extractSegmentStrings(geom)
        // only request intersection finder if there are segments (ie NOT for point inputs)
        if (lineSegStr.size > 0) {
            val segsIntersect: Boolean = prepLine.intersectionFinder!!.intersects(lineSegStr)
            // MD - performance testing
            //		boolean segsIntersect = false;
            if (segsIntersect) return true
        }
        /**
         * For L/L case we are done
         */
        if (geom.dimension == 1) return false
        /**
         * For L/A case, need to check for proper inclusion of the target in the test
         */
        if (geom.dimension == 2
            && prepLine.isAnyTargetComponentInTest(geom)
        ) return true
        /**
         * For L/P case, need to check if any points lie on line(s)
         */
        return if (geom.dimension == 0) isAnyTestPointInTarget(geom) else false
    }

    /**
     * Tests whether any representative point of the test Geometry intersects
     * the target geometry.
     * Only handles test geometries which are Puntal (dimension 0)
     *
     * @param geom a Puntal geometry to test
     * @return true if any point of the argument intersects the prepared geometry
     */
    protected fun isAnyTestPointInTarget(testGeom: Geometry?): Boolean {
        /**
         * This could be optimized by using the segment index on the lineal target.
         * However, it seems like the L/P case would be pretty rare in practice.
         */
        val locator = PointLocator()
        val coords: MutableList<Coordinate> = ComponentCoordinateExtracter.getCoordinates(testGeom!!)
        val i: Iterator<*> = coords.iterator()
        while (i.hasNext()) {
            val p = i.next() as Coordinate
            if (locator.intersects(p, prepLine.geometry)) return true
        }
        return false
    }

    companion object {
        /**
         * Computes the intersects predicate between a [PreparedLineString]
         * and a [Geometry].
         *
         * @param prep the prepared linestring
         * @param geom a test geometry
         * @return true if the linestring intersects the geometry
         */
        fun intersects(prep: PreparedLineString, geom: Geometry): Boolean {
            val op = PreparedLineStringIntersects(prep)
            return op.intersects(geom)
        }
    }
}