/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.geom

import org.locationtech.jts.geom.util.GeometryEditor

/**
 * An interface for classes which process the coordinates in a [CoordinateSequence].
 * A filter can either record information about each coordinate,
 * or change the value of the coordinate.
 * Filters can be
 * used to implement operations such as coordinate transformations, centroid and
 * envelope computation, and many other functions.
 * [Geometry] classes support the concept of applying a
 * `CoordinateSequenceFilter` to each
 * [CoordinateSequence]s they contain.
 *
 * For maximum efficiency, the execution of filters can be short-circuited by using the [.isDone] method.
 *
 * `CoordinateSequenceFilter` is
 * an example of the Gang-of-Four Visitor pattern.
 *
 * **Note**: In general, it is preferable to treat Geometrys as immutable.
 * Mutation should be performed by creating a new Geometry object (see [GeometryEditor]
 * and [GeometryTransformer] for convenient ways to do this).
 * An exception to this rule is when a new Geometry has been created via [Geometry.copy].
 * In this case mutating the Geometry will not cause aliasing issues,
 * and a filter is a convenient way to implement coordinate transformation.
 *
 * @see Geometry.apply
 * @see GeometryTransformer
 *
 * @see GeometryEditor
 * @see Geometry.apply
 * @author Martin Davis
 * @version 1.7
 */
interface CoordinateSequenceFilter {
    /**
     * Performs an operation on a coordinate in a [CoordinateSequence].
     *
     * @param seq  the `CoordinateSequence` to which the filter is applied
     * @param i the index of the coordinate to apply the filter to
     */
    fun filter(seq: CoordinateSequence?, i: Int)

    /**
     * Reports whether the application of this filter can be terminated.
     * Once this method returns <tt>true</tt>, it must
     * continue to return <tt>true</tt> on every subsequent call.
     *
     * @return true if the application of this filter can be terminated.
     */
    val isDone: Boolean

    /**
     * Reports whether the execution of this filter
     * has modified the coordinates of the geometry.
     * If so, [Geometry.geometryChanged] will be executed
     * after this filter has finished being executed.
     *
     * Most filters can simply return a constant value reflecting
     * whether they are able to change the coordinates.
     *
     * @return true if this filter has changed the coordinates of the geometry
     */
    val isGeometryChanged: Boolean
}