/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.operation.overlay

import org.locationtech.jts.algorithm.PointLocation
import org.locationtech.jts.geom.*
import org.locationtech.jts.geomgraph.DirectedEdge
import org.locationtech.jts.geomgraph.EdgeRing
import org.locationtech.jts.geomgraph.PlanarGraph
import org.locationtech.jts.util.Assert

/**
 * Forms [Polygon]s out of a graph of [DirectedEdge]s.
 * The edges to use are marked as being in the result Area.
 *
 *
 * @version 1.7
 */
class PolygonBuilder(private val geometryFactory: GeometryFactory) {
    private val shellList: MutableList<EdgeRing> = ArrayList()

    /**
     * Add a complete graph.
     * The graph is assumed to contain one or more polygons,
     * possibly with holes.
     */
    fun add(graph: PlanarGraph) {
        add(graph.getEdgeEnds(), graph.getNodes())
    }

    /**
     * Add a set of edges and nodes, which form a graph.
     * The graph is assumed to contain one or more polygons,
     * possibly with holes.
     */
    fun add(dirEdges: Collection<*>, nodes: Collection<*>?) {
        PlanarGraph.linkResultDirectedEdges(nodes!!)
        val maxEdgeRings: MutableList<MaximalEdgeRing> = buildMaximalEdgeRings(dirEdges)
        val freeHoleList: MutableList<EdgeRing> = ArrayList()
        val edgeRings: MutableList<EdgeRing> = buildMinimalEdgeRings(maxEdgeRings, shellList, freeHoleList)
        sortShellsAndHoles(edgeRings, shellList, freeHoleList)
        placeFreeHoles(shellList, freeHoleList)
        //Assert: every hole on freeHoleList has a shell assigned to it
    }

    val polygons: MutableList<Polygon>
        get() = computePolygons(shellList)

    /**
     * for all DirectedEdges in result, form them into MaximalEdgeRings
     */
    private fun buildMaximalEdgeRings(dirEdges: Collection<*>): MutableList<MaximalEdgeRing> {
        val maxEdgeRings: MutableList<MaximalEdgeRing> = ArrayList()
        val it = dirEdges.iterator()
        while (it.hasNext()) {
            val de = it.next() as DirectedEdge
            if (de.isInResult && de.label!!.isArea()) {
                // if this edge has not yet been processed
                if (de.edgeRing == null) {
                    val er: MaximalEdgeRing =
                        MaximalEdgeRing(de, geometryFactory)
                    maxEdgeRings.add(er)
                    er.setInResult()
                    //System.out.println("max node degree = " + er.getMaxDegree());
                }
            }
        }
        return maxEdgeRings
    }

    private fun buildMinimalEdgeRings(
        maxEdgeRings: MutableList<MaximalEdgeRing>,
        shellList: MutableList<EdgeRing>,
        freeHoleList: MutableList<EdgeRing>
    ): MutableList<EdgeRing> {
        val edgeRings: MutableList<EdgeRing> = ArrayList()
        val it: Iterator<*> = maxEdgeRings.iterator()
        while (it.hasNext()) {
            val er: MaximalEdgeRing =
                it.next() as MaximalEdgeRing
            if (er.getMaxNodeDegree() > 2) {
                er.linkDirectedEdgesForMinimalEdgeRings()
                val minEdgeRings: MutableList<EdgeRing> = er.buildMinimalRings()
                // at this point we can go ahead and attempt to place holes, if this EdgeRing is a polygon
                val shell = findShell(minEdgeRings)
                if (shell != null) {
                    placePolygonHoles(shell, minEdgeRings)
                    shellList.add(shell)
                } else {
                    freeHoleList.addAll(minEdgeRings)
                }
            } else {
                edgeRings.add(er)
            }
        }
        return edgeRings
    }

    /**
     * This method takes a list of MinimalEdgeRings derived from a MaximalEdgeRing,
     * and tests whether they form a Polygon.  This is the case if there is a single shell
     * in the list.  In this case the shell is returned.
     * The other possibility is that they are a series of connected holes, in which case
     * no shell is returned.
     *
     * @return the shell EdgeRing, if there is one
     * or null, if all the rings are holes
     */
    private fun findShell(minEdgeRings: MutableList<EdgeRing>): EdgeRing? {
        var shellCount = 0
        var shell: EdgeRing? = null
        val it: Iterator<*> = minEdgeRings.iterator()
        while (it.hasNext()) {
            val er: EdgeRing = it.next() as MinimalEdgeRing
            if (!er.isHole()) {
                shell = er
                shellCount++
            }
        }
        Assert.isTrue(shellCount <= 1, "found two shells in MinimalEdgeRing list")
        return shell
    }

    /**
     * This method assigns the holes for a Polygon (formed from a list of
     * MinimalEdgeRings) to its shell.
     * Determining the holes for a MinimalEdgeRing polygon serves two purposes:
     *
     *  * it is faster than using a point-in-polygon check later on.
     *  * it ensures correctness, since if the PIP test was used the point
     * chosen might lie on the shell, which might return an incorrect result from the
     * PIP test
     *
     */
    private fun placePolygonHoles(shell: EdgeRing, minEdgeRings: MutableList<EdgeRing>) {
        val it: Iterator<*> = minEdgeRings.iterator()
        while (it.hasNext()) {
            val er: MinimalEdgeRing =
                it.next() as MinimalEdgeRing
            if (er.isHole()) {
                er.setShell(shell)
            }
        }
    }

    /**
     * For all rings in the input list,
     * determine whether the ring is a shell or a hole
     * and add it to the appropriate list.
     * Due to the way the DirectedEdges were linked,
     * a ring is a shell if it is oriented CW, a hole otherwise.
     */
    private fun sortShellsAndHoles(edgeRings: MutableList<EdgeRing>, shellList: MutableList<EdgeRing>, freeHoleList: MutableList<EdgeRing>) {
        val it: Iterator<*> = edgeRings.iterator()
        while (it.hasNext()) {
            val er = it.next() as EdgeRing
            //      er.setInResult();
            if (er.isHole()) {
                freeHoleList.add(er)
            } else {
                shellList.add(er)
            }
        }
    }

    /**
     * This method determines finds a containing shell for all holes
     * which have not yet been assigned to a shell.
     * These "free" holes should
     * all be **properly** contained in their parent shells, so it is safe to use the
     * `findEdgeRingContaining` method.
     * (This is the case because any holes which are NOT
     * properly contained (i.e. are connected to their
     * parent shell) would have formed part of a MaximalEdgeRing
     * and been handled in a previous step).
     *
     * @throws TopologyException if a hole cannot be assigned to a shell
     */
    private fun placeFreeHoles(shellList: MutableList<EdgeRing>, freeHoleList: MutableList<*>) {
        val it: Iterator<*> = freeHoleList.iterator()
        while (it.hasNext()) {
            val hole = it.next() as EdgeRing
            // only place this hole if it doesn't yet have a shell
            if (hole.getShell() == null) {
                val shell = findEdgeRingContaining(hole, shellList)
                    ?: throw TopologyException("unable to assign hole to a shell", hole.getCoordinate(0))
                //        Assert.isTrue(shell != null, "unable to assign hole to a shell");
                hole.setShell(shell)
            }
        }
    }

    private fun computePolygons(shellList: MutableList<EdgeRing>): MutableList<Polygon> {
        val resultPolyList: MutableList<Polygon> = ArrayList()
        // add Polygons for all shells
        val it: Iterator<*> = shellList.iterator()
        while (it.hasNext()) {
            val er = it.next() as EdgeRing
            val poly = er.toPolygon(geometryFactory)
            resultPolyList.add(poly)
        }
        return resultPolyList
    }

    companion object {
        /**
         * Find the innermost enclosing shell EdgeRing containing the argument EdgeRing, if any.
         * The innermost enclosing ring is the *smallest* enclosing ring.
         * The algorithm used depends on the fact that:
         * <br></br>
         * ring A contains ring B if envelope(ring A) contains envelope(ring B)
         * <br></br>
         * This routine is only safe to use if the chosen point of the hole
         * is known to be properly contained in a shell
         * (which is guaranteed to be the case if the hole does not touch its shell)
         *
         * @return containing EdgeRing, if there is one
         * or null if no containing EdgeRing is found
         */
        private fun findEdgeRingContaining(testEr: EdgeRing, shellList: MutableList<EdgeRing>): EdgeRing? {
            val testRing = testEr.getLinearRing()
            val testEnv = testRing!!.envelopeInternal
            var testPt: Coordinate? = testRing.getCoordinateN(0)
            var minShell: EdgeRing? = null
            var minShellEnv: Envelope? = null
            val it: Iterator<*> = shellList.iterator()
            while (it.hasNext()) {
                val tryShell = it.next() as EdgeRing
                val tryShellRing = tryShell.getLinearRing()
                val tryShellEnv = tryShellRing!!.envelopeInternal
                // the hole envelope cannot equal the shell envelope
                // (also guards against testing rings against themselves)
                if (tryShellEnv == testEnv) {
                    continue
                }
                // hole must be contained in shell
                if (!tryShellEnv.contains(testEnv)) {
                    continue
                }
                testPt = CoordinateArrays.ptNotInList(testRing.coordinates, tryShellRing.coordinates)
                var isContained = false
                if (PointLocation.isInRing(testPt!!, tryShellRing.coordinates)) isContained = true

                // check if this new containing ring is smaller than the current minimum ring
                if (isContained) {
                    if (minShell == null
                        || minShellEnv!!.contains(tryShellEnv)
                    ) {
                        minShell = tryShell
                        minShellEnv = minShell.getLinearRing()!!.envelopeInternal
                    }
                }
            }
            return minShell
        }
    }
}