/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.planargraph

/**
 * The base class for all graph component classes.
 * Maintains flags of use in generic graph algorithms.
 * Provides two flags:
 *
 *  * **marked** - typically this is used to indicate a state that persists
 * for the course of the graph's lifetime.  For instance, it can be
 * used to indicate that a component has been logically deleted from the graph.
 *  * **visited** - this is used to indicate that a component has been processed
 * or visited by an single graph algorithm.  For instance, a breadth-first traversal of the
 * graph might use this to indicate that a node has already been traversed.
 * The visited flag may be set and cleared many times during the lifetime of a graph.
 *
 *
 * Graph components support storing user context data.  This will typically be
 * used by client algorithms which use planar graphs.
 *
 * @version 1.7
 */
abstract class GraphComponent {
    /**
     * Tests if a component has been marked at some point during the processing
     * involving this graph.
     * @return `true` if the component has been marked
     */
    /**
     * Sets the marked flag for this component.
     * @param isMarked the desired value of the marked flag
     */
    var isMarked = false
    /**
     * Tests if a component has been visited during the course of a graph algorithm
     * @return `true` if the component has been visited
     */
    /**
     * Sets the visited flag for this component.
     * @param isVisited the desired value of the visited flag
     */
    var isVisited = false
    /**
     * Gets the user-defined data for this component.
     *
     * @return the user-defined data
     */
    /**
     * Sets the user-defined data for this component.
     *
     * @param data an Object containing user-defined data
     */
    /**
     * Gets the user-defined data for this component.
     *
     * @return the user-defined data
     */
    /**
     * Sets the user-defined data for this component.
     *
     * @param data an Object containing user-defined data
     */
    var context: Any? = null
        get() = field
        set(data) {
            field = data
        }

    /**
     * Tests whether this component has been removed from its containing graph
     *
     * @return `true` if this component is removed
     */
    abstract val isRemoved: Boolean

    companion object {
        /**
         * Sets the Visited state for all [GraphComponent]s in an [Iterator]
         *
         * @param i the Iterator to scan
         * @param visited the state to set the visited flag to
         */
        fun setVisited(i: Iterator<*>, visited: Boolean) {
            while (i.hasNext()) {
                val comp = i.next() as GraphComponent
                comp.isVisited = visited
            }
        }

        /**
         * Sets the Marked state for all [GraphComponent]s in an [Iterator]
         *
         * @param i the Iterator to scan
         * @param marked the state to set the Marked flag to
         */
        fun setMarked(i: Iterator<*>, marked: Boolean) {
            while (i.hasNext()) {
                val comp = i.next() as GraphComponent
                comp.isMarked = marked
            }
        }

        /**
         * Finds the first [GraphComponent] in a [Iterator] set
         * which has the specified visited state.
         *
         * @param i an Iterator of GraphComponents
         * @param visitedState the visited state to test
         * @return the first component found, or `null` if none found
         */
        fun getComponentWithVisitedState(i: Iterator<*>, visitedState: Boolean): GraphComponent? {
            while (i.hasNext()) {
                val comp = i.next() as GraphComponent
                if (comp.isVisited == visitedState) return comp
            }
            return null
        }
    }
}