package com.macrofocus.application.root

import java.awt.Component
import java.awt.Container
import java.awt.Dimension
import java.awt.LayoutManager
import java.io.Serializable


/**
 * A vertical flow layout is similar to a flow layuot but it layouts the
 * components vertically instead of horizontally.
 */
class VerticalFlowLayout @JvmOverloads constructor(
    halign: Int = CENTER, valign: Int = CENTER,
    /**
     * Sets the Hgap attribute of the VerticalFlowLayout object
     *
     * @param  hgap  The new Hgap value
     */
    var hgap: Int = 5,
    /**
     * Sets the Vgap attribute of the VerticalFlowLayout object
     *
     * @param  vgap  The new Vgap value
     */
    var vgap: Int = 5
) :
    LayoutManager, Serializable {
    /**
     * Gets the Halignment attribute of the VerticalFlowLayout object
     *
     * @return    The Halignment value
     */
    var halignment = 0

    /**
     * Gets the Valignment attribute of the VerticalFlowLayout object
     *
     * @return    The Valignment value
     */
    var valignment = 0
    /**
     * Gets the Hgap attribute of the VerticalFlowLayout object
     *
     * @return    The Hgap value
     */
    /**
     * Gets the Vgap attribute of the VerticalFlowLayout object
     *
     * @return    The Vgap value
     */

    /**
     * Sets the Alignment attribute of the VerticalFlowLayout object
     *
     * @param  halign  The new Alignment value
     * @param  valign  The new Alignment value
     */
    fun setAlignment(halign: Int, valign: Int) {
        halignment = halign
        valignment = valign
    }

    /**
     * Adds a feature to the LayoutComponent attribute of the VerticalFlowLayout
     * object
     *
     * @param  name  The feature to be added to the LayoutComponent attribute
     * @param  comp  The feature to be added to the LayoutComponent attribute
     */
    override fun addLayoutComponent(name: String, comp: Component) {}

    /**
     * Description of the Method
     *
     * @param  comp  Description of Parameter
     */
    override fun removeLayoutComponent(comp: Component) {}

    /**
     * Description of the Method
     *
     * @param  target  Description of Parameter
     * @return         Description of the Returned Value
     */
    override fun preferredLayoutSize(target: Container): Dimension {
        synchronized(target.treeLock) {
            val dim = Dimension(0, 0)
            val nmembers = target.componentCount
            var firstVisibleComponent = true
            for (ii in 0 until nmembers) {
                val m = target.getComponent(ii)
                if (m.isVisible) {
                    val d = m.preferredSize
                    dim.width = Math.max(dim.width, d.width)
                    if (firstVisibleComponent) {
                        firstVisibleComponent = false
                    } else {
                        dim.height += vgap
                    }
                    dim.height += d.height
                }
            }
            val insets = target.insets
            dim.width += insets.left + insets.right + hgap * 2
            dim.height += insets.top + insets.bottom + vgap * 2
            return dim
        }
    }

    /**
     * Description of the Method
     *
     * @param  target  Description of Parameter
     * @return         Description of the Returned Value
     */
    override fun minimumLayoutSize(target: Container): Dimension {
        synchronized(target.treeLock) {
            val dim = Dimension(0, 0)
            val nmembers = target.componentCount
            var firstVisibleComponent = true
            for (ii in 0 until nmembers) {
                val m = target.getComponent(ii)
                if (m.isVisible) {
                    val d = m.preferredSize
                    dim.width = Math.max(dim.width, d.width)
                    if (firstVisibleComponent) {
                        firstVisibleComponent = false
                    } else {
                        dim.height += vgap
                    }
                    dim.height += d.height
                }
            }
            val insets = target.insets
            dim.width += insets.left + insets.right + hgap * 2
            dim.height += insets.top + insets.bottom + vgap * 2
            return dim
        }
    }

    /**
     * Description of the Method
     *
     * @param  target  Description of Parameter
     */
    override fun layoutContainer(target: Container) {
        synchronized(target.treeLock) {
            val insets = target.insets
            val maxheight = target.height - (insets.top + insets.bottom + vgap * 2)
            val nmembers = target.componentCount
            var y = 0
            val preferredSize = preferredLayoutSize(target)
            val targetSize = target.size
            when (valignment) {
                TOP -> y = insets.top
                CENTER -> y = (targetSize.height - preferredSize.height) / 2
                BOTTOM -> y = targetSize.height - preferredSize.height - insets.bottom
            }
            for (i in 0 until nmembers) {
                val m = target.getComponent(i)
                if (m.isVisible) {
                    val d = m.preferredSize
                    m.setSize(d.width, d.height)
                    if (y + d.height <= maxheight) {
                        if (y > 0) {
                            y += vgap
                        }
                        var x = 0
                        when (halignment) {
                            LEFT -> x = insets.left
                            CENTER -> x = (targetSize.width - d.width) / 2
                            RIGHT -> x = targetSize.width - d.width - insets.right
                        }
                        m.setLocation(x, y)
                        y += d.getHeight().toInt()
                    } else {
                        break
                    }
                }
            }
        }
    }

    /** {@inheritDoc}  */
    override fun toString(): String {
        var halign = ""
        when (halignment) {
            TOP -> halign = "top"
            CENTER -> halign = "center"
            BOTTOM -> halign = "bottom"
        }
        var valign = ""
        when (valignment) {
            TOP -> valign = "top"
            CENTER -> valign = "center"
            BOTTOM -> valign = "bottom"
        }
        return javaClass.name + "[hgap=" + hgap + ",vgap=" + vgap + ",halign=" + halign + ",valign=" + valign + "]"
    }

    companion object {
        /**
         * Description of the Field
         */
        const val TOP = 0

        /**
         * Description of the Field
         */
        const val CENTER = 1

        /**
         * Description of the Field
         */
        const val BOTTOM = 2

        /**
         * Description of the Field
         */
        const val LEFT = 3

        /**
         * Description of the Field
         */
        const val RIGHT = 4
    }
    /**
     * Constructor for the VerticalFlowLayout object
     *
     * @param  halign  Description of Parameter
     * @param  valign  Description of Parameter
     * @param  hgap    Description of Parameter
     * @param  vgap    Description of Parameter
     */
    /**
     * Constructor for the VerticalFlowLayout object
     */
    /**
     * Constructor for the VerticalFlowLayout object
     *
     * @param  halign  Description of Parameter
     * @param  valign  Description of Parameter
     */
    init {
        vgap = vgap
        setAlignment(halign, valign)
    }
}