package com.macrofocus.application.file

import java.io.File
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream
import java.util.regex.Pattern

object FileHelper {
    fun getSuffix(f: File?): String? {
        return if (f != null) {
            val s = f.path
            getSuffix(s)
        } else {
            null
        }
    }

    fun getSuffix(s: String): String? {
        var s = s
        var suffix: String? = null
        var i = s.lastIndexOf('.')
        if (i > 0 && i < s.length - 1) {
            suffix = s.substring(i + 1).lowercase()
            s = s.substring(0, i)
        }
        if (suffix != null && suffix == "gz") {
            i = s.lastIndexOf('.')
            if (i > 0 && i < s.length - 1) {
                suffix = s.substring(i + 1).lowercase() + "." + suffix
            }
        }
        return suffix
    }

    fun hasSuffix(file: File, extension: String): Boolean {
        val name = file.name
        return name.endsWith(".$extension")
    }

    fun replaceSuffix(file: File?, suffix: String): File? {
        val parent = if (file != null && file.parentFile != null) file.parentFile else null
        val name = if (file != null) file.name else "Untitled"
        return if (name.endsWith(".$suffix")) {
            file
        } else {
            val index = name.lastIndexOf(".")
            if (index > 0 && name.length - index <= 5) {
                val newName = name.substring(0, index)
                File(parent, "$newName.$suffix")
            } else {
                File(parent, "$name.$suffix")
            }
        }
    }

    fun addSuffix(name: String?, suffix: String): String {
        var name = name
        name = name ?: "Untitled"
        return if (name.endsWith(".$suffix")) {
            name
        } else {
            "$name.$suffix"
        }
    }

    fun replaceSuffix(name: String?, suffix: String): String {
        var name = name
        name = name ?: "Untitled"
        return if (name.endsWith(".$suffix")) {
            name
        } else {
            val index = name.lastIndexOf(".")
            if (index > 0) {
                val newName = name.substring(0, index)
                "$newName.$suffix"
            } else {
                "$name.$suffix"
            }
        }
    }

    fun removeSuffix(file: File, suffix: String): String {
        val name = file.name
        return if (name.endsWith(".$suffix")) {
            val index = name.lastIndexOf(".")
            name.substring(0, index)
        } else {
            name
        }
    }

    fun removeSuffix(file: File): String {
        val name = file.name
        return removeSuffix(name)
    }

    fun removeSuffix(name: String): String {
        return if (name.contains(".")) {
            val index = name.lastIndexOf(".")
            name.substring(0, index)
        } else {
            name
        }
    }

    fun getName(path: String): String {
        return if (path.contains("\\") || path.contains("/")) {
            val index = Math.max(path.lastIndexOf("\\"), path.lastIndexOf("/"))
            path.substring(index + 1)
        } else {
            path
        }
    }

    fun getPath(path: String): String {
        return if (path.contains("\\") || path.contains("/")) {
            val index = Math.max(path.lastIndexOf("\\"), path.lastIndexOf("/"))
            path.substring(0, index)
        } else {
            path
        }
    }

    @Throws(IOException::class)
    fun copy(`in`: InputStream, out: OutputStream) {
        // do not allow other threads to read from the
        // input or write to the output while copying is
        // taking place
        synchronized(`in`) {
            synchronized(out) {
                val buffer = ByteArray(256)
                while (true) {
                    val bytesRead = `in`.read(buffer)
                    if (bytesRead == -1) break
                    out.write(buffer, 0, bytesRead)
                }
            }
        }
    }

    /**
     * Get the relative path from one file to another, specifying the directory separator.
     * If one of the provided resources does not exist, it is assumed to be a file unless it ends with '/' or
     * '\'.
     *
     * @param targetPath is calculated to this file
     * @param basePath is calculated from this file
     * @param pathSeparator directory separator. The platform default is not assumed so that we can test Unix behaviour when running on Windows (for example)
     * @return target relative path from the base
     */
    fun getRelativePath(targetPath: String, basePath: String, pathSeparator: String): String {
        var targetPath = targetPath
        var basePath = basePath
        if (targetPath.lowercase().startsWith("file:/")) {
            targetPath = targetPath.substring(6)
            while (targetPath.startsWith("/")) targetPath = targetPath.substring(1)
        }
        if (basePath.lowercase().startsWith("file:/")) {
            basePath = basePath.substring(6)
            while (basePath.startsWith("/")) basePath = basePath.substring(1)
        }
        var tempS = basePath
        while (true) {
            val baseF = File(tempS)
            if (baseF.exists() && baseF.isFile) {
                basePath = baseF.absolutePath
                break
            }
            if (pathSeparator != "/") break
            if (tempS.startsWith("/")) break
            tempS = "/$tempS"
        }
        tempS = targetPath
        while (true) {
            val targetF = File(tempS)
            if (targetF.exists() && targetF.isFile) {
                targetPath = targetF.absolutePath
                break
            }
            if (pathSeparator != "/") break
            if (tempS.startsWith("/")) break
            tempS = "/$tempS"
        }

        // Normalize the paths
        //System.out.println("ResourceUtils.getRelativePath()..target("+pathSeparator+"):"+targetPath);
        //System.out.println("ResourceUtils.getRelativePath()..base("+pathSeparator+"):"+basePath);
        var normalizedTargetPath: String? = FilenameUtils.normalizeNoEndSeparator(targetPath)
        var normalizedBasePath: String? = FilenameUtils.normalizeNoEndSeparator(basePath)
        // Undo the changes to the separators made by normalization
        if (pathSeparator == "/") {
            normalizedTargetPath = FilenameUtils.separatorsToUnix(normalizedTargetPath)
            normalizedBasePath = FilenameUtils.separatorsToUnix(normalizedBasePath)
        } else if (pathSeparator == "\\") {
            normalizedTargetPath = FilenameUtils.separatorsToWindows(normalizedTargetPath)
            normalizedBasePath = FilenameUtils.separatorsToWindows(normalizedBasePath)
        } else {
            throw IllegalArgumentException("Unrecognised dir separator '$pathSeparator'")
        }

        //System.out.println("ResourceUtils.getRelativePath()..normalizedTarget("+pathSeparator+"):"+normalizedTargetPath);
        //System.out.println("ResourceUtils.getRelativePath()..normalizedBase("+pathSeparator+"):"+normalizedBasePath);
        val base = normalizedBasePath!!.split(Pattern.quote(pathSeparator).toRegex()).toTypedArray()
        val target = normalizedTargetPath!!.split(Pattern.quote(pathSeparator).toRegex()).toTypedArray()
        // First get all the common elements. Store them as a string,
        // and also count how many of them there are.
        val common = StringBuffer()
        var commonIndex = 0
        while (commonIndex < target.size && commonIndex < base.size && target[commonIndex].trim { it <= ' ' } == base[commonIndex].trim { it <= ' ' }) {
            common.append(target[commonIndex].trim { it <= ' ' } + pathSeparator)
            commonIndex++
        }
        if (commonIndex == 0) {
            // No single common path element. This most
            // likely indicates differing drive letters, like C: and D:.
            // These paths cannot be relativized.
            val ff = File(targetPath)
            return if (ff.exists() && ff.isFile) ff.absolutePath else throw IllegalArgumentException("No common path element found for '$normalizedTargetPath' and '$normalizedBasePath'")
        }
        // The number of directories we have to backtrack depends on whether the base is a file or a dir
        // For example, the relative path from
        //
        // /foo/bar/baz/gg/ff to /foo/bar/baz
        //
        // ".." if ff is a file
        // "../.." if ff is a directory
        //
        // The following is a heuristic to figure out if the base refers to a file or dir. It's not perfect, because
        // the resource referred to by this path may not actually exist, but it's the best I can do
        var baseIsFile = true
        val baseResource = File(normalizedBasePath)
        if (baseResource.exists()) {
            baseIsFile = baseResource.isFile
        } else if (basePath.endsWith(pathSeparator)) {
            baseIsFile = false
        }
        val relative = StringBuffer()
        if (base.size != commonIndex) {
            val numDirsUp = if (baseIsFile) base.size - commonIndex - 1 else base.size - commonIndex
            for (i in 0 until numDirsUp) {
                relative.append("..$pathSeparator")
            }
        }
        if (common.length >= normalizedTargetPath.length) {
            return "."
        }
        relative.append(normalizedTargetPath.substring(common.length))
        //System.out.println("ResourceUtils.getRelativePath()..relativeTarget:"+relative.toString());
        return relative.toString()
    }
}
