package com.macrofocus.application.file

import java.io.File
import java.io.IOException
import java.util.*
import java.util.regex.Matcher
import java.util.regex.Pattern

/**
 * General file name and file path manipulation utilities.
 *
 *
 * When dealing with file names you can hit problems when moving from a Windows
 * based development machine to a Unix based production machine.
 * This class aims to help avoid those problems.
 *
 *
 * **NOTE**: You may be able to avoid using this class entirely simply by
 * using JDK [File][java.io.File] objects and the two argument constructor
 * [File(File,String)][java.io.File.File].
 *
 *
 * Most methods on this class are designed to work the same on both Unix and Windows.
 * Those that don't include 'System', 'Unix' or 'Windows' in their name.
 *
 *
 * Most methods recognise both separators (forward and back), and both
 * sets of prefixes. See the javadoc of each method for details.
 *
 *
 * This class defines six components within a file name
 * (example C:\dev\project\file.txt):
 *
 *  * the prefix - C:\
 *  * the path - dev\project\
 *  * the full path - C:\dev\project\
 *  * the name - file.txt
 *  * the base name - file
 *  * the extension - txt
 *
 * Note that this class works best if directory file names end with a separator.
 * If you omit the last separator, it is impossible to determine if the file name
 * corresponds to a file or a directory. As a result, we have chosen to say
 * it corresponds to a file.
 *
 *
 * This class only supports Unix and Windows style names.
 * Prefixes are matched as follows:
 * <pre>
 * Windows:
 * a\b\c.txt           --&gt; ""          --&gt; relative
 * \a\b\c.txt          --&gt; "\"         --&gt; current drive absolute
 * C:a\b\c.txt         --&gt; "C:"        --&gt; drive relative
 * C:\a\b\c.txt        --&gt; "C:\"       --&gt; absolute
 * \\server\a\b\c.txt  --&gt; "\\server\" --&gt; UNC
 *
 * Unix:
 * a/b/c.txt           --&gt; ""          --&gt; relative
 * /a/b/c.txt          --&gt; "/"         --&gt; absolute
 * ~/a/b/c.txt         --&gt; "~/"        --&gt; current user
 * ~                   --&gt; "~/"        --&gt; current user (slash added)
 * ~user/a/b/c.txt     --&gt; "~user/"    --&gt; named user
 * ~user               --&gt; "~user/"    --&gt; named user (slash added)
</pre> *
 * Both prefix styles are matched always, irrespective of the machine that you are
 * currently running on.
 *
 *
 * Origin of code: Excalibur, Alexandria, Tomcat, Commons-Utils.
 *
 * @since 1.1
 */
object FilenameUtils {
    private val EMPTY_STRING_ARRAY: Array<String?> = arrayOfNulls(0)
    private val EMPTY_STRING: String = ""
    private val NOT_FOUND: Int = -1

    /**
     * The extension separator character.
     * @since 1.4
     */
    val EXTENSION_SEPARATOR: Char = '.'

    /**
     * The extension separator String.
     * @since 1.4
     */
    val EXTENSION_SEPARATOR_STR: String = Character.toString(EXTENSION_SEPARATOR)

    /**
     * The Unix separator character.
     */
    private val UNIX_SEPARATOR: Char = '/'

    /**
     * The Windows separator character.
     */
    private val WINDOWS_SEPARATOR: Char = '\\'

    /**
     * The system separator character.
     */
    private val SYSTEM_SEPARATOR: Char = File.separatorChar

    /**
     * The separator character that is the opposite of the system separator.
     */
    private var OTHER_SEPARATOR: Char = 0.toChar()
    //-----------------------------------------------------------------------
    /**
     * Determines if Windows file system is in use.
     *
     * @return true if the system is Windows
     */
    val isSystemWindows: Boolean
        get() {
            return SYSTEM_SEPARATOR == WINDOWS_SEPARATOR
        }
    //-----------------------------------------------------------------------
    /**
     * Checks if the character is a separator.
     *
     * @param ch  the character to check
     * @return true if it is a separator character
     */
    private fun isSeparator(ch: Char): Boolean {
        return ch == UNIX_SEPARATOR || ch == WINDOWS_SEPARATOR
    }
    //-----------------------------------------------------------------------
    /**
     * Normalizes a path, removing double and single dot path steps.
     *
     *
     * This method normalizes a path to a standard format.
     * The input may contain separators in either Unix or Windows format.
     * The output will contain separators in the format of the system.
     *
     *
     * A trailing slash will be retained.
     * A double slash will be merged to a single slash (but UNC names are handled).
     * A single dot path segment will be removed.
     * A double dot will cause that path segment and the one before to be removed.
     * If the double dot has no parent path segment to work with, `null`
     * is returned.
     *
     *
     * The output will be the same on both Unix and Windows except
     * for the separator character.
     * <pre>
     * /foo//               --&gt;   /foo/
     * /foo/./              --&gt;   /foo/
     * /foo/../bar          --&gt;   /bar
     * /foo/../bar/         --&gt;   /bar/
     * /foo/../bar/../baz   --&gt;   /baz
     * //foo//./bar         --&gt;   /foo/bar
     * /../                 --&gt;   null
     * ../foo               --&gt;   null
     * foo/bar/..           --&gt;   foo/
     * foo/../../bar        --&gt;   null
     * foo/../bar           --&gt;   bar
     * //server/foo/../bar  --&gt;   //server/bar
     * //server/../bar      --&gt;   null
     * C:\foo\..\bar        --&gt;   C:\bar
     * C:\..\bar            --&gt;   null
     * ~/foo/../bar/        --&gt;   ~/bar/
     * ~/../bar             --&gt;   null
    </pre> *
     * (Note the file separator returned will be correct for Windows/Unix)
     *
     * @param fileName  the fileName to normalize, null returns null
     * @return the normalized fileName, or null if invalid. Null bytes inside string will be removed
     */
    fun normalize(fileName: String?): String? {
        return doNormalize(fileName, SYSTEM_SEPARATOR, true)
    }

    /**
     * Normalizes a path, removing double and single dot path steps.
     *
     *
     * This method normalizes a path to a standard format.
     * The input may contain separators in either Unix or Windows format.
     * The output will contain separators in the format specified.
     *
     *
     * A trailing slash will be retained.
     * A double slash will be merged to a single slash (but UNC names are handled).
     * A single dot path segment will be removed.
     * A double dot will cause that path segment and the one before to be removed.
     * If the double dot has no parent path segment to work with, `null`
     * is returned.
     *
     *
     * The output will be the same on both Unix and Windows except
     * for the separator character.
     * <pre>
     * /foo//               --&gt;   /foo/
     * /foo/./              --&gt;   /foo/
     * /foo/../bar          --&gt;   /bar
     * /foo/../bar/         --&gt;   /bar/
     * /foo/../bar/../baz   --&gt;   /baz
     * //foo//./bar         --&gt;   /foo/bar
     * /../                 --&gt;   null
     * ../foo               --&gt;   null
     * foo/bar/..           --&gt;   foo/
     * foo/../../bar        --&gt;   null
     * foo/../bar           --&gt;   bar
     * //server/foo/../bar  --&gt;   //server/bar
     * //server/../bar      --&gt;   null
     * C:\foo\..\bar        --&gt;   C:\bar
     * C:\..\bar            --&gt;   null
     * ~/foo/../bar/        --&gt;   ~/bar/
     * ~/../bar             --&gt;   null
    </pre> *
     * The output will be the same on both Unix and Windows including
     * the separator character.
     *
     * @param fileName  the fileName to normalize, null returns null
     * @param unixSeparator `true` if a unix separator should
     * be used or `false` if a windows separator should be used.
     * @return the normalized fileName, or null if invalid. Null bytes inside string will be removed
     * @since 2.0
     */
    fun normalize(fileName: String?, unixSeparator: Boolean): String? {
        val separator: Char = if (unixSeparator) UNIX_SEPARATOR else WINDOWS_SEPARATOR
        return doNormalize(fileName, separator, true)
    }
    //-----------------------------------------------------------------------
    /**
     * Normalizes a path, removing double and single dot path steps,
     * and removing any final directory separator.
     *
     *
     * This method normalizes a path to a standard format.
     * The input may contain separators in either Unix or Windows format.
     * The output will contain separators in the format of the system.
     *
     *
     * A trailing slash will be removed.
     * A double slash will be merged to a single slash (but UNC names are handled).
     * A single dot path segment will be removed.
     * A double dot will cause that path segment and the one before to be removed.
     * If the double dot has no parent path segment to work with, `null`
     * is returned.
     *
     *
     * The output will be the same on both Unix and Windows except
     * for the separator character.
     * <pre>
     * /foo//               --&gt;   /foo
     * /foo/./              --&gt;   /foo
     * /foo/../bar          --&gt;   /bar
     * /foo/../bar/         --&gt;   /bar
     * /foo/../bar/../baz   --&gt;   /baz
     * //foo//./bar         --&gt;   /foo/bar
     * /../                 --&gt;   null
     * ../foo               --&gt;   null
     * foo/bar/..           --&gt;   foo
     * foo/../../bar        --&gt;   null
     * foo/../bar           --&gt;   bar
     * //server/foo/../bar  --&gt;   //server/bar
     * //server/../bar      --&gt;   null
     * C:\foo\..\bar        --&gt;   C:\bar
     * C:\..\bar            --&gt;   null
     * ~/foo/../bar/        --&gt;   ~/bar
     * ~/../bar             --&gt;   null
    </pre> *
     * (Note the file separator returned will be correct for Windows/Unix)
     *
     * @param fileName  the fileName to normalize, null returns null
     * @return the normalized fileName, or null if invalid. Null bytes inside string will be removed
     */
    fun normalizeNoEndSeparator(fileName: String?): String? {
        return doNormalize(fileName, SYSTEM_SEPARATOR, false)
    }

    /**
     * Normalizes a path, removing double and single dot path steps,
     * and removing any final directory separator.
     *
     *
     * This method normalizes a path to a standard format.
     * The input may contain separators in either Unix or Windows format.
     * The output will contain separators in the format specified.
     *
     *
     * A trailing slash will be removed.
     * A double slash will be merged to a single slash (but UNC names are handled).
     * A single dot path segment will be removed.
     * A double dot will cause that path segment and the one before to be removed.
     * If the double dot has no parent path segment to work with, `null`
     * is returned.
     *
     *
     * The output will be the same on both Unix and Windows including
     * the separator character.
     * <pre>
     * /foo//               --&gt;   /foo
     * /foo/./              --&gt;   /foo
     * /foo/../bar          --&gt;   /bar
     * /foo/../bar/         --&gt;   /bar
     * /foo/../bar/../baz   --&gt;   /baz
     * //foo//./bar         --&gt;   /foo/bar
     * /../                 --&gt;   null
     * ../foo               --&gt;   null
     * foo/bar/..           --&gt;   foo
     * foo/../../bar        --&gt;   null
     * foo/../bar           --&gt;   bar
     * //server/foo/../bar  --&gt;   //server/bar
     * //server/../bar      --&gt;   null
     * C:\foo\..\bar        --&gt;   C:\bar
     * C:\..\bar            --&gt;   null
     * ~/foo/../bar/        --&gt;   ~/bar
     * ~/../bar             --&gt;   null
    </pre> *
     *
     * @param fileName  the fileName to normalize, null returns null
     * @param unixSeparator `true` if a unix separator should
     * be used or `false` if a windows separator should be used.
     * @return the normalized fileName, or null if invalid. Null bytes inside string will be removed
     * @since 2.0
     */
    fun normalizeNoEndSeparator(fileName: String?, unixSeparator: Boolean): String? {
        val separator: Char = if (unixSeparator) UNIX_SEPARATOR else WINDOWS_SEPARATOR
        return doNormalize(fileName, separator, false)
    }

    /**
     * Internal method to perform the normalization.
     *
     * @param fileName  the fileName
     * @param separator The separator character to use
     * @param keepSeparator  true to keep the final separator
     * @return the normalized fileName. Null bytes inside string will be removed.
     */
    private fun doNormalize(fileName: String?, separator: Char, keepSeparator: Boolean): String? {
        if (fileName == null) {
            return null
        }
        failIfNullBytePresent(fileName)
        var size: Int = fileName.length
        if (size == 0) {
            return fileName
        }
        val prefix: Int = getPrefixLength(fileName)
        if (prefix < 0) {
            return null
        }
        val array: CharArray = CharArray(size + 2) // +1 for possible extra slash, +2 for arraycopy
        fileName.toCharArray(array, 0, 0, fileName.length)

        // fix separators throughout
        val otherSeparator: Char = if (separator == SYSTEM_SEPARATOR) OTHER_SEPARATOR else SYSTEM_SEPARATOR
        for (i in array.indices) {
            if (array.get(i) == otherSeparator) {
                array.set(i, separator)
            }
        }

        // add extra separator on the end to simplify code below
        var lastIsDirectory: Boolean = true
        if (array.get(size - 1) != separator) {
            array.set(size++, separator)
            lastIsDirectory = false
        }

        // adjoining slashes
        run {
            var i: Int = prefix + 1
            while (i < size) {
                if (array.get(i) == separator && array.get(i - 1) == separator) {
                    System.arraycopy(array, i, array, i - 1, size - i)
                    size--
                    i--
                }
                i++
            }
        }

        // dot slash
        var i: Int = prefix + 1
        while (i < size) {
            if (((array.get(i) == separator) && (array.get(i - 1) == '.') &&
                        (i == prefix + 1 || array.get(i - 2) == separator))
            ) {
                if (i == size - 1) {
                    lastIsDirectory = true
                }
                System.arraycopy(array, i + 1, array, i - 1, size - i)
                size -= 2
                i--
            }
            i++
        }
        i = prefix + 2
        outer@ while (i < size) {
            if (((array.get(i) == separator) && (array.get(i - 1) == '.') && (array.get(i - 2) == '.') &&
                        (i == prefix + 2 || array.get(i - 3) == separator))
            ) {
                if (i == prefix + 2) {
                    return null
                }
                if (i == size - 1) {
                    lastIsDirectory = true
                }
                var j: Int
                j = i - 4
                while (j >= prefix) {
                    if (array.get(j) == separator) {
                        // remove b/../ from a/b/../c
                        System.arraycopy(array, i + 1, array, j + 1, size - i)
                        size -= i - j
                        i = j + 1
                        i++
                        continue@outer
                    }
                    j--
                }
                // remove a/../ from a/../c
                System.arraycopy(array, i + 1, array, prefix, size - i)
                size -= i + 1 - prefix
                i = prefix + 1
            }
            i++
        }
        if (size <= 0) {  // should never be less than 0
            return EMPTY_STRING
        }
        if (size <= prefix) {  // should never be less than prefix
            return String(array, 0, size)
        }
        if (lastIsDirectory && keepSeparator) {
            return String(array, 0, size) // keep trailing separator
        }
        return String(array, 0, size - 1) // lose trailing separator
    }
    //-----------------------------------------------------------------------
    /**
     * Concatenates a fileName to a base path using normal command line style rules.
     *
     *
     * The effect is equivalent to resultant directory after changing
     * directory to the first argument, followed by changing directory to
     * the second argument.
     *
     *
     * The first argument is the base path, the second is the path to concatenate.
     * The returned path is always normalized via [.normalize],
     * thus `..` is handled.
     *
     *
     * If `pathToAdd` is absolute (has an absolute prefix), then
     * it will be normalized and returned.
     * Otherwise, the paths will be joined, normalized and returned.
     *
     *
     * The output will be the same on both Unix and Windows except
     * for the separator character.
     * <pre>
     * /foo/      + bar        --&gt;  /foo/bar
     * /foo       + bar        --&gt;  /foo/bar
     * /foo       + /bar       --&gt;  /bar
     * /foo       + C:/bar     --&gt;  C:/bar
     * /foo       + C:bar      --&gt;  C:bar (*)
     * /foo/a/    + ../bar     --&gt;  /foo/bar
     * /foo/      + ../../bar  --&gt;  null
     * /foo/      + /bar       --&gt;  /bar
     * /foo/..    + /bar       --&gt;  /bar
     * /foo       + bar/c.txt  --&gt;  /foo/bar/c.txt
     * /foo/c.txt + bar        --&gt;  /foo/c.txt/bar (!)
    </pre> *
     * (*) Note that the Windows relative drive prefix is unreliable when
     * used with this method.
     * (!) Note that the first parameter must be a path. If it ends with a name, then
     * the name will be built into the concatenated path. If this might be a problem,
     * use [.getFullPath] on the base path argument.
     *
     * @param basePath  the base path to attach to, always treated as a path
     * @param fullFileNameToAdd  the fileName (or path) to attach to the base
     * @return the concatenated path, or null if invalid.  Null bytes inside string will be removed
     */
    fun concat(basePath: String?, fullFileNameToAdd: String): String? {
        val prefix: Int = getPrefixLength(fullFileNameToAdd)
        if (prefix < 0) {
            return null
        }
        if (prefix > 0) {
            return normalize(fullFileNameToAdd)
        }
        if (basePath == null) {
            return null
        }
        val len: Int = basePath.length
        if (len == 0) {
            return normalize(fullFileNameToAdd)
        }
        val ch: Char = basePath.get(len - 1)
        if (isSeparator(ch)) {
            return normalize(basePath + fullFileNameToAdd)
        }
        return normalize(basePath + '/' + fullFileNameToAdd)
    }

    /**
     * Determines whether the `parent` directory contains the `child` element (a file or directory).
     *
     *
     * The files names are expected to be normalized.
     *
     *
     * Edge cases:
     *
     *  * A `directory` must not be null: if null, throw IllegalArgumentException
     *  * A directory does not contain itself: return false
     *  * A null child file is not contained in any parent: return false
     *
     *
     * @param canonicalParent
     * the file to consider as the parent.
     * @param canonicalChild
     * the file to consider as the child.
     * @return true is the candidate leaf is under by the specified composite. False otherwise.
     * @throws IOException
     * if an IO error occurs while checking the files.
     * @since 2.2
     * @see FileUtils.directoryContains
     */
//    @Throws(IOException::class)
//    fun directoryContains(canonicalParent: String?, canonicalChild: String?): Boolean {
//
//        // Fail fast against NullPointerException
//        if (canonicalParent == null) {
//            throw IllegalArgumentException("Directory must not be null")
//        }
//        if (canonicalChild == null) {
//            return false
//        }
//        if (IOCase.SYSTEM.checkEquals(canonicalParent, canonicalChild)) {
//            return false
//        }
//        return IOCase.SYSTEM.checkStartsWith(canonicalChild, canonicalParent)
//    }
    //-----------------------------------------------------------------------
    /**
     * Converts all separators to the Unix separator of forward slash.
     *
     * @param path  the path to be changed, null ignored
     * @return the updated path
     */
    fun separatorsToUnix(path: String?): String? {
        if (path == null || path.indexOf(WINDOWS_SEPARATOR) == NOT_FOUND) {
            return path
        }
        return path.replace(WINDOWS_SEPARATOR, UNIX_SEPARATOR)
    }

    /**
     * Converts all separators to the Windows separator of backslash.
     *
     * @param path  the path to be changed, null ignored
     * @return the updated path
     */
    fun separatorsToWindows(path: String?): String? {
        if (path == null || path.indexOf(UNIX_SEPARATOR) == NOT_FOUND) {
            return path
        }
        return path.replace(UNIX_SEPARATOR, WINDOWS_SEPARATOR)
    }

    /**
     * Converts all separators to the system separator.
     *
     * @param path  the path to be changed, null ignored
     * @return the updated path
     */
    fun separatorsToSystem(path: String?): String? {
        if (path == null) {
            return null
        }
        return if (isSystemWindows) separatorsToWindows(path) else separatorsToUnix(path)
    }
    //-----------------------------------------------------------------------
    /**
     * Returns the length of the fileName prefix, such as `C:/` or `~/`.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     *
     *
     * The prefix length includes the first slash in the full fileName
     * if applicable. Thus, it is possible that the length returned is greater
     * than the length of the input string.
     * <pre>
     * Windows:
     * a\b\c.txt           --&gt; ""          --&gt; relative
     * \a\b\c.txt          --&gt; "\"         --&gt; current drive absolute
     * C:a\b\c.txt         --&gt; "C:"        --&gt; drive relative
     * C:\a\b\c.txt        --&gt; "C:\"       --&gt; absolute
     * \\server\a\b\c.txt  --&gt; "\\server\" --&gt; UNC
     * \\\a\b\c.txt        --&gt;  error, length = -1
     *
     * Unix:
     * a/b/c.txt           --&gt; ""          --&gt; relative
     * /a/b/c.txt          --&gt; "/"         --&gt; absolute
     * ~/a/b/c.txt         --&gt; "~/"        --&gt; current user
     * ~                   --&gt; "~/"        --&gt; current user (slash added)
     * ~user/a/b/c.txt     --&gt; "~user/"    --&gt; named user
     * ~user               --&gt; "~user/"    --&gt; named user (slash added)
     * //server/a/b/c.txt  --&gt; "//server/"
     * ///a/b/c.txt        --&gt; error, length = -1
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     * ie. both Unix and Windows prefixes are matched regardless.
     *
     * Note that a leading // (or \\) is used to indicate a UNC name on Windows.
     * These must be followed by a server name, so double-slashes are not collapsed
     * to a single slash at the start of the fileName.
     *
     * @param fileName  the fileName to find the prefix in, null returns -1
     * @return the length of the prefix, -1 if invalid or null
     */
    fun getPrefixLength(fileName: String?): Int {
        if (fileName == null) {
            return NOT_FOUND
        }
        val len: Int = fileName.length
        if (len == 0) {
            return 0
        }
        var ch0: Char = fileName.get(0)
        if (ch0 == ':') {
            return NOT_FOUND
        }
        if (len == 1) {
            if (ch0 == '~') {
                return 2 // return a length greater than the input
            }
            return if (isSeparator(ch0)) 1 else 0
        }
        if (ch0 == '~') {
            var posUnix: Int = fileName.indexOf(UNIX_SEPARATOR, 1)
            var posWin: Int = fileName.indexOf(WINDOWS_SEPARATOR, 1)
            if (posUnix == NOT_FOUND && posWin == NOT_FOUND) {
                return len + 1 // return a length greater than the input
            }
            posUnix = if (posUnix == NOT_FOUND) posWin else posUnix
            posWin = if (posWin == NOT_FOUND) posUnix else posWin
            return Math.min(posUnix, posWin) + 1
        }
        val ch1: Char = fileName.get(1)
        if (ch1 == ':') {
            ch0 = Character.toUpperCase(ch0)
            if (ch0 >= 'A' && ch0 <= 'Z') {
                if (len == 2 || isSeparator(fileName.get(2)) == false) {
                    return 2
                }
                return 3
            } else if (ch0 == UNIX_SEPARATOR) {
                return 1
            }
            return NOT_FOUND
//        } else if (isSeparator(ch0) && isSeparator(ch1)) {
//            var posUnix: Int = fileName.indexOf(UNIX_SEPARATOR, 2)
//            var posWin: Int = fileName.indexOf(WINDOWS_SEPARATOR, 2)
//            if ((posUnix == NOT_FOUND && posWin == NOT_FOUND) || (posUnix == 2) || (posWin == 2)) {
//                return NOT_FOUND
//            }
//            posUnix = if (posUnix == NOT_FOUND) posWin else posUnix
//            posWin = if (posWin == NOT_FOUND) posUnix else posWin
//            val pos: Int = Math.min(posUnix, posWin) + 1
//            val hostnamePart: String = fileName.substring(2, pos - 1)
//            return if (isValidHostName(hostnamePart)) pos else NOT_FOUND
        } else {
            return if (isSeparator(ch0)) 1 else 0
        }
    }

    /**
     * Returns the index of the last directory separator character.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The position of the last forward or backslash is returned.
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to find the last path separator in, null returns -1
     * @return the index of the last separator character, or -1 if there
     * is no such character
     */
    fun indexOfLastSeparator(fileName: String?): Int {
        if (fileName == null) {
            return NOT_FOUND
        }
        val lastUnixPos: Int = fileName.lastIndexOf(UNIX_SEPARATOR)
        val lastWindowsPos: Int = fileName.lastIndexOf(WINDOWS_SEPARATOR)
        return Math.max(lastUnixPos, lastWindowsPos)
    }

    /**
     * Returns the index of the last extension separator character, which is a dot.
     *
     *
     * This method also checks that there is no directory separator after the last dot. To do this it uses
     * [.indexOfLastSeparator] which will handle a file in either Unix or Windows format.
     *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on, with the
     * exception of a possible [IllegalArgumentException] on Windows (see below).
     *
     * **Note:** This method used to have a hidden problem for names like "foo.exe:bar.txt".
     * In this case, the name wouldn't be the name of a file, but the identifier of an
     * alternate data stream (bar.txt) on the file foo.exe. The method used to return
     * ".txt" here, which would be misleading. Commons IO 2.7, and later versions, are throwing
     * an [IllegalArgumentException] for names like this.
     *
     * @param fileName
     * the fileName to find the last extension separator in, null returns -1
     * @return the index of the last extension separator character, or -1 if there is no such character
     * @throws IllegalArgumentException **Windows only:** The fileName parameter is, in fact,
     * the identifier of an Alternate Data Stream, for example "foo.exe:bar.txt".
     */
    @Throws(IllegalArgumentException::class)
    fun indexOfExtension(fileName: String?): Int {
        if (fileName == null) {
            return NOT_FOUND
        }
        if (isSystemWindows) {
            // Special handling for NTFS ADS: Don't accept colon in the fileName.
            val offset: Int = fileName.indexOf(':', getAdsCriticalOffset(fileName))
            if (offset != -1) {
                throw IllegalArgumentException("NTFS ADS separator (':') in file name is forbidden.")
            }
        }
        val extensionPos: Int = fileName.lastIndexOf(EXTENSION_SEPARATOR)
        val lastSeparator: Int = indexOfLastSeparator(fileName)
        return if (lastSeparator > extensionPos) NOT_FOUND else extensionPos
    }
    //-----------------------------------------------------------------------
    /**
     * Gets the prefix from a full fileName, such as `C:/`
     * or `~/`.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The prefix includes the first slash in the full fileName where applicable.
     * <pre>
     * Windows:
     * a\b\c.txt           --&gt; ""          --&gt; relative
     * \a\b\c.txt          --&gt; "\"         --&gt; current drive absolute
     * C:a\b\c.txt         --&gt; "C:"        --&gt; drive relative
     * C:\a\b\c.txt        --&gt; "C:\"       --&gt; absolute
     * \\server\a\b\c.txt  --&gt; "\\server\" --&gt; UNC
     *
     * Unix:
     * a/b/c.txt           --&gt; ""          --&gt; relative
     * /a/b/c.txt          --&gt; "/"         --&gt; absolute
     * ~/a/b/c.txt         --&gt; "~/"        --&gt; current user
     * ~                   --&gt; "~/"        --&gt; current user (slash added)
     * ~user/a/b/c.txt     --&gt; "~user/"    --&gt; named user
     * ~user               --&gt; "~user/"    --&gt; named user (slash added)
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     * ie. both Unix and Windows prefixes are matched regardless.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the prefix of the file, null if invalid. Null bytes inside string will be removed
     */
    fun getPrefix(fileName: String?): String? {
        if (fileName == null) {
            return null
        }
        val len: Int = getPrefixLength(fileName)
        if (len < 0) {
            return null
        }
        if (len > fileName.length) {
            failIfNullBytePresent(fileName + UNIX_SEPARATOR)
            return fileName + UNIX_SEPARATOR
        }
        val path: String = fileName.substring(0, len)
        failIfNullBytePresent(path)
        return path
    }

    /**
     * Gets the path from a full fileName, which excludes the prefix.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The method is entirely text based, and returns the text before and
     * including the last forward or backslash.
     * <pre>
     * C:\a\b\c.txt --&gt; a\b\
     * ~/a/b/c.txt  --&gt; a/b/
     * a.txt        --&gt; ""
     * a/b/c        --&gt; a/b/
     * a/b/c/       --&gt; a/b/c/
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     *
     * This method drops the prefix from the result.
     * See [.getFullPath] for the method that retains the prefix.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the path of the file, an empty string if none exists, null if invalid.
     * Null bytes inside string will be removed
     */
    fun getPath(fileName: String?): String? {
        return doGetPath(fileName, 1)
    }

    /**
     * Gets the path from a full fileName, which excludes the prefix, and
     * also excluding the final directory separator.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The method is entirely text based, and returns the text before the
     * last forward or backslash.
     * <pre>
     * C:\a\b\c.txt --&gt; a\b
     * ~/a/b/c.txt  --&gt; a/b
     * a.txt        --&gt; ""
     * a/b/c        --&gt; a/b
     * a/b/c/       --&gt; a/b/c
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     *
     * This method drops the prefix from the result.
     * See [.getFullPathNoEndSeparator] for the method that retains the prefix.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the path of the file, an empty string if none exists, null if invalid.
     * Null bytes inside string will be removed
     */
    fun getPathNoEndSeparator(fileName: String?): String? {
        return doGetPath(fileName, 0)
    }

    /**
     * Does the work of getting the path.
     *
     * @param fileName  the fileName
     * @param separatorAdd  0 to omit the end separator, 1 to return it
     * @return the path. Null bytes inside string will be removed
     */
    private fun doGetPath(fileName: String?, separatorAdd: Int): String? {
        if (fileName == null) {
            return null
        }
        val prefix: Int = getPrefixLength(fileName)
        if (prefix < 0) {
            return null
        }
        val index: Int = indexOfLastSeparator(fileName)
        val endIndex: Int = index + separatorAdd
        if ((prefix >= fileName.length) || (index < 0) || (prefix >= endIndex)) {
            return EMPTY_STRING
        }
        val path: String = fileName.substring(prefix, endIndex)
        failIfNullBytePresent(path)
        return path
    }

    /**
     * Gets the full path from a full fileName, which is the prefix + path.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The method is entirely text based, and returns the text before and
     * including the last forward or backslash.
     * <pre>
     * C:\a\b\c.txt --&gt; C:\a\b\
     * ~/a/b/c.txt  --&gt; ~/a/b/
     * a.txt        --&gt; ""
     * a/b/c        --&gt; a/b/
     * a/b/c/       --&gt; a/b/c/
     * C:           --&gt; C:
     * C:\          --&gt; C:\
     * ~            --&gt; ~/
     * ~/           --&gt; ~/
     * ~user        --&gt; ~user/
     * ~user/       --&gt; ~user/
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the path of the file, an empty string if none exists, null if invalid
     */
    fun getFullPath(fileName: String?): String? {
        return doGetFullPath(fileName, true)
    }

    /**
     * Gets the full path from a full fileName, which is the prefix + path,
     * and also excluding the final directory separator.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The method is entirely text based, and returns the text before the
     * last forward or backslash.
     * <pre>
     * C:\a\b\c.txt --&gt; C:\a\b
     * ~/a/b/c.txt  --&gt; ~/a/b
     * a.txt        --&gt; ""
     * a/b/c        --&gt; a/b
     * a/b/c/       --&gt; a/b/c
     * C:           --&gt; C:
     * C:\          --&gt; C:\
     * ~            --&gt; ~
     * ~/           --&gt; ~
     * ~user        --&gt; ~user
     * ~user/       --&gt; ~user
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the path of the file, an empty string if none exists, null if invalid
     */
    fun getFullPathNoEndSeparator(fileName: String?): String? {
        return doGetFullPath(fileName, false)
    }

    /**
     * Does the work of getting the path.
     *
     * @param fileName  the fileName
     * @param includeSeparator  true to include the end separator
     * @return the path
     */
    private fun doGetFullPath(fileName: String?, includeSeparator: Boolean): String? {
        if (fileName == null) {
            return null
        }
        val prefix: Int = getPrefixLength(fileName)
        if (prefix < 0) {
            return null
        }
        if (prefix >= fileName.length) {
            if (includeSeparator) {
                return getPrefix(fileName) // add end slash if necessary
            }
            return fileName
        }
        val index: Int = indexOfLastSeparator(fileName)
        if (index < 0) {
            return fileName.substring(0, prefix)
        }
        var end: Int = index + (if (includeSeparator) 1 else 0)
        if (end == 0) {
            end++
        }
        return fileName.substring(0, end)
    }

    /**
     * Gets the name minus the path from a full fileName.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The text after the last forward or backslash is returned.
     * <pre>
     * a/b/c.txt --&gt; c.txt
     * a.txt     --&gt; a.txt
     * a/b/c     --&gt; c
     * a/b/c/    --&gt; ""
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the name of the file without the path, or an empty string if none exists.
     * Null bytes inside string will be removed
     */
    fun getName(fileName: String?): String? {
        if (fileName == null) {
            return null
        }
        failIfNullBytePresent(fileName)
        val index: Int = indexOfLastSeparator(fileName)
        return fileName.substring(index + 1)
    }

    /**
     * Check the input for null bytes, a sign of unsanitized data being passed to to file level functions.
     *
     * This may be used for poison byte attacks.
     * @param path the path to check
     */
    private fun failIfNullBytePresent(path: String) {
        val len: Int = path.length
        for (i in 0 until len) {
            if (path.get(i).toInt() == 0) {
                throw IllegalArgumentException(
                    "Null byte present in file/path name. There are no " +
                            "known legitimate use cases for such data, but several injection attacks may use it"
                )
            }
        }
    }

    /**
     * Gets the base name, minus the full path and extension, from a full fileName.
     *
     *
     * This method will handle a file in either Unix or Windows format.
     * The text after the last forward or backslash and before the last dot is returned.
     * <pre>
     * a/b/c.txt --&gt; c
     * a.txt     --&gt; a
     * a/b/c     --&gt; c
     * a/b/c/    --&gt; ""
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the name of the file without the path, or an empty string if none exists. Null bytes inside string
     * will be removed
     */
    fun getBaseName(fileName: String?): String? {
        return removeExtension(getName(fileName))
    }

    /**
     * Gets the extension of a fileName.
     *
     *
     * This method returns the textual part of the fileName after the last dot.
     * There must be no directory separator after the dot.
     * <pre>
     * foo.txt      --&gt; "txt"
     * a/b/c.jpg    --&gt; "jpg"
     * a/b.txt/c    --&gt; ""
     * a/b/c        --&gt; ""
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on, with the
     * exception of a possible [IllegalArgumentException] on Windows (see below).
     *
     *
     *
     * **Note:** This method used to have a hidden problem for names like "foo.exe:bar.txt".
     * In this case, the name wouldn't be the name of a file, but the identifier of an
     * alternate data stream (bar.txt) on the file foo.exe. The method used to return
     * ".txt" here, which would be misleading. Commons IO 2.7, and later versions, are throwing
     * an [IllegalArgumentException] for names like this.
     *
     * @param fileName the fileName to retrieve the extension of.
     * @return the extension of the file or an empty string if none exists or `null`
     * if the fileName is `null`.
     * @throws IllegalArgumentException **Windows only:** The fileName parameter is, in fact,
     * the identifier of an Alternate Data Stream, for example "foo.exe:bar.txt".
     */
    @Throws(IllegalArgumentException::class)
    fun getExtension(fileName: String?): String? {
        if (fileName == null) {
            return null
        }
        val index: Int = indexOfExtension(fileName)
        if (index == NOT_FOUND) {
            return EMPTY_STRING
        }
        return fileName.substring(index + 1)
    }

    /**
     * Special handling for NTFS ADS: Don't accept colon in the fileName.
     *
     * @param fileName a file name
     * @return ADS offsets.
     */
    private fun getAdsCriticalOffset(fileName: String): Int {
        // Step 1: Remove leading path segments.
        val offset1: Int = fileName.lastIndexOf(SYSTEM_SEPARATOR)
        val offset2: Int = fileName.lastIndexOf(OTHER_SEPARATOR)
        if (offset1 == -1) {
            if (offset2 == -1) {
                return 0
            }
            return offset2 + 1
        }
        if (offset2 == -1) {
            return offset1 + 1
        }
        return Math.max(offset1, offset2) + 1
    }
    //-----------------------------------------------------------------------
    /**
     * Removes the extension from a fileName.
     *
     *
     * This method returns the textual part of the fileName before the last dot.
     * There must be no directory separator after the dot.
     * <pre>
     * foo.txt    --&gt; foo
     * a\b\c.jpg  --&gt; a\b\c
     * a\b\c      --&gt; a\b\c
     * a.b\c      --&gt; a.b\c
    </pre> *
     *
     *
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param fileName  the fileName to query, null returns null
     * @return the fileName minus the extension
     */
    fun removeExtension(fileName: String?): String? {
        if (fileName == null) {
            return null
        }
        failIfNullBytePresent(fileName)
        val index: Int = indexOfExtension(fileName)
        if (index == NOT_FOUND) {
            return fileName
        }
        return fileName.substring(0, index)
    }

    /**
     * Checks whether two fileNames are equal using the case rules of the system.
     *
     *
     * No processing is performed on the fileNames other than comparison.
     * The check is case-sensitive on Unix and case-insensitive on Windows.
     *
     * @param fileName1  the first fileName to query, may be null
     * @param fileName2  the second fileName to query, may be null
     * @return true if the fileNames are equal, null equals null
     * @see IOCase.SYSTEM
     */
//    fun equalsOnSystem(fileName1: String?, fileName2: String?): Boolean {
//        return equals(fileName1, fileName2, false, IOCase.SYSTEM)
//    }
    //-----------------------------------------------------------------------
    /**
     * Checks whether two fileNames are equal after both have been normalized.
     *
     *
     * Both fileNames are first passed to [.normalize].
     * The check is then performed in a case-sensitive manner.
     *
     * @param fileName1  the first fileName to query, may be null
     * @param fileName2  the second fileName to query, may be null
     * @return true if the fileNames are equal, null equals null
     * @see IOCase.SENSITIVE
     */
//    fun equalsNormalized(fileName1: String?, fileName2: String?): Boolean {
//        return equals(fileName1, fileName2, true, IOCase.SENSITIVE)
//    }

    /**
     * Checks whether two fileNames are equal after both have been normalized
     * and using the case rules of the system.
     *
     *
     * Both fileNames are first passed to [.normalize].
     * The check is then performed case-sensitive on Unix and
     * case-insensitive on Windows.
     *
     * @param fileName1  the first fileName to query, may be null
     * @param fileName2  the second fileName to query, may be null
     * @return true if the fileNames are equal, null equals null
     * @see IOCase.SYSTEM
     */
//    fun equalsNormalizedOnSystem(fileName1: String?, fileName2: String?): Boolean {
//        return equals(fileName1, fileName2, true, IOCase.SYSTEM)
//    }
    /**
     * Checks whether two fileNames are equal, optionally normalizing and providing
     * control over the case-sensitivity.
     *
     * @param fileName1  the first fileName to query, may be null
     * @param fileName2  the second fileName to query, may be null
     * @param normalized  whether to normalize the fileNames
     * @param caseSensitivity  what case sensitivity rule to use, null means case-sensitive
     * @return true if the fileNames are equal, null equals null
     * @since 1.3
     */
    //-----------------------------------------------------------------------
    /**
     * Checks whether two fileNames are equal exactly.
     *
     *
     * No processing is performed on the fileNames other than comparison,
     * thus this is merely a null-safe case-sensitive equals.
     *
     * @param fileName1  the first fileName to query, may be null
     * @param fileName2  the second fileName to query, may be null
     * @return true if the fileNames are equal, null equals null
     * @see IOCase.SENSITIVE
     */
    @JvmOverloads
//    fun equals(
//        fileName1: String?, fileName2: String?,
//        normalized: Boolean = false, caseSensitivity: IOCase? = IOCase.SENSITIVE
//    ): Boolean {
//        var fileName1: String? = fileName1
//        var fileName2: String? = fileName2
//        var caseSensitivity: IOCase? = caseSensitivity
//        if (fileName1 == null || fileName2 == null) {
//            return fileName1 == null && fileName2 == null
//        }
//        if (normalized) {
//            fileName1 = normalize(fileName1)
//            fileName2 = normalize(fileName2)
//            Objects.requireNonNull(fileName1, "Error normalizing one or both of the file names")
//            Objects.requireNonNull(fileName2, "Error normalizing one or both of the file names")
//        }
//        if (caseSensitivity == null) {
//            caseSensitivity = IOCase.SENSITIVE
//        }
//        return caseSensitivity.checkEquals(fileName1, fileName2)
//    }
    //-----------------------------------------------------------------------
    /**
     * Checks whether the extension of the fileName is that specified.
     *
     *
     * This method obtains the extension as the textual part of the fileName
     * after the last dot. There must be no directory separator after the dot.
     * The extension check is case-sensitive on all platforms.
     *
     * @param fileName  the fileName to query, null returns false
     * @param extension  the extension to check for, null or empty checks for no extension
     * @return true if the fileName has the specified extension
     * @throws java.lang.IllegalArgumentException if the supplied fileName contains null bytes
     */
    fun isExtension(fileName: String?, extension: String?): Boolean {
        if (fileName == null) {
            return false
        }
        failIfNullBytePresent(fileName)
        if (extension == null || extension.isEmpty()) {
            return indexOfExtension(fileName) == NOT_FOUND
        }
        val fileExt: String? = getExtension(fileName)
        return (fileExt == extension)
    }

    /**
     * Checks whether the extension of the fileName is one of those specified.
     *
     *
     * This method obtains the extension as the textual part of the fileName
     * after the last dot. There must be no directory separator after the dot.
     * The extension check is case-sensitive on all platforms.
     *
     * @param fileName  the fileName to query, null returns false
     * @param extensions  the extensions to check for, null checks for no extension
     * @return true if the fileName is one of the extensions
     * @throws java.lang.IllegalArgumentException if the supplied fileName contains null bytes
     */
    fun isExtension(fileName: String?, vararg extensions: String): Boolean {
        if (fileName == null) {
            return false
        }
        failIfNullBytePresent(fileName)
        if (extensions == null || extensions.size == 0) {
            return indexOfExtension(fileName) == NOT_FOUND
        }
        val fileExt: String? = getExtension(fileName)
        for (extension: String in extensions) {
            if ((fileExt == extension)) {
                return true
            }
        }
        return false
    }

    /**
     * Checks whether the extension of the fileName is one of those specified.
     *
     *
     * This method obtains the extension as the textual part of the fileName
     * after the last dot. There must be no directory separator after the dot.
     * The extension check is case-sensitive on all platforms.
     *
     * @param fileName  the fileName to query, null returns false
     * @param extensions  the extensions to check for, null checks for no extension
     * @return true if the fileName is one of the extensions
     * @throws java.lang.IllegalArgumentException if the supplied fileName contains null bytes
     */
    fun isExtension(fileName: String?, extensions: Collection<String>?): Boolean {
        if (fileName == null) {
            return false
        }
        failIfNullBytePresent(fileName)
        if (extensions == null || extensions.isEmpty()) {
            return indexOfExtension(fileName) == NOT_FOUND
        }
        val fileExt: String? = getExtension(fileName)
        for (extension: String in extensions) {
            if ((fileExt == extension)) {
                return true
            }
        }
        return false
    }

    /**
     * Checks a fileName to see if it matches the specified wildcard matcher
     * using the case rules of the system.
     *
     *
     * The wildcard matcher uses the characters '?' and '*' to represent a
     * single or multiple (zero or more) wildcard characters.
     * This is the same as often found on Dos/Unix command lines.
     * The check is case-sensitive on Unix and case-insensitive on Windows.
     * <pre>
     * wildcardMatch("c.txt", "*.txt")      --&gt; true
     * wildcardMatch("c.txt", "*.jpg")      --&gt; false
     * wildcardMatch("a/b/c.txt", "a/b/ *")  --&gt; true
     * wildcardMatch("c.txt", "*.???")      --&gt; true
     * wildcardMatch("c.txt", "*.????")     --&gt; false
    </pre> *
     * N.B. the sequence "*?" does not work properly at present in match strings.
     *
     * @param fileName  the fileName to match on
     * @param wildcardMatcher  the wildcard string to match against
     * @return true if the fileName matches the wildcard string
     * @see IOCase.SYSTEM
     */
//    fun wildcardMatchOnSystem(fileName: String?, wildcardMatcher: String?): Boolean {
//        return wildcardMatch(fileName, wildcardMatcher, IOCase.SYSTEM)
//    }
    /**
     * Checks a fileName to see if it matches the specified wildcard matcher
     * allowing control over case-sensitivity.
     *
     *
     * The wildcard matcher uses the characters '?' and '*' to represent a
     * single or multiple (zero or more) wildcard characters.
     * N.B. the sequence "*?" does not work properly at present in match strings.
     *
     * @param fileName  the fileName to match on
     * @param wildcardMatcher  the wildcard string to match against
     * @param caseSensitivity  what case sensitivity rule to use, null means case-sensitive
     * @return true if the fileName matches the wildcard string
     * @since 1.3
     */
    //-----------------------------------------------------------------------
    /**
     * Checks a fileName to see if it matches the specified wildcard matcher,
     * always testing case-sensitive.
     *
     *
     * The wildcard matcher uses the characters '?' and '*' to represent a
     * single or multiple (zero or more) wildcard characters.
     * This is the same as often found on Dos/Unix command lines.
     * The check is case-sensitive always.
     * <pre>
     * wildcardMatch("c.txt", "*.txt")      --&gt; true
     * wildcardMatch("c.txt", "*.jpg")      --&gt; false
     * wildcardMatch("a/b/c.txt", "a/b/ *")  --&gt; true
     * wildcardMatch("c.txt", "*.???")      --&gt; true
     * wildcardMatch("c.txt", "*.????")     --&gt; false
    </pre> *
     * N.B. the sequence "*?" does not work properly at present in match strings.
     *
     * @param fileName  the fileName to match on
     * @param wildcardMatcher  the wildcard string to match against
     * @return true if the fileName matches the wildcard string
     * @see IOCase.SENSITIVE
     */
//    @JvmOverloads
//    fun wildcardMatch(
//        fileName: String?,
//        wildcardMatcher: String?,
//        caseSensitivity: IOCase? = IOCase.SENSITIVE
//    ): Boolean {
//        var caseSensitivity: IOCase? = caseSensitivity
//        if (fileName == null && wildcardMatcher == null) {
//            return true
//        }
//        if (fileName == null || wildcardMatcher == null) {
//            return false
//        }
//        if (caseSensitivity == null) {
//            caseSensitivity = IOCase.SENSITIVE
//        }
//        val wcs: Array<String?> = splitOnTokens(wildcardMatcher)
//        var anyChars: Boolean = false
//        var textIdx: Int = 0
//        var wcsIdx: Int = 0
//        val backtrack: Deque<IntArray> = ArrayDeque(wcs.size)
//
//        // loop around a backtrack stack, to handle complex * matching
//        do {
//            if (!backtrack.isEmpty()) {
//                val array: IntArray = backtrack.pop()
//                wcsIdx = array.get(0)
//                textIdx = array.get(1)
//                anyChars = true
//            }
//
//            // loop whilst tokens and text left to process
//            while (wcsIdx < wcs.size) {
//                if ((wcs.get(wcsIdx) == "?")) {
//                    // ? so move to next text char
//                    textIdx++
//                    if (textIdx > fileName.length) {
//                        break
//                    }
//                    anyChars = false
//                } else if ((wcs.get(wcsIdx) == "*")) {
//                    // set any chars status
//                    anyChars = true
//                    if (wcsIdx == wcs.size - 1) {
//                        textIdx = fileName.length
//                    }
//                } else {
//                    // matching text token
//                    if (anyChars) {
//                        // any chars then try to locate text token
//                        textIdx = caseSensitivity.checkIndexOf(fileName, textIdx, wcs.get(wcsIdx))
//                        if (textIdx == NOT_FOUND) {
//                            // token not found
//                            break
//                        }
//                        val repeat: Int = caseSensitivity.checkIndexOf(fileName, textIdx + 1, wcs.get(wcsIdx))
//                        if (repeat >= 0) {
//                            backtrack.push(intArrayOf(wcsIdx, repeat))
//                        }
//                    } else {
//                        // matching from current position
//                        if (!caseSensitivity.checkRegionMatches(fileName, textIdx, wcs.get(wcsIdx))) {
//                            // couldnt match token
//                            break
//                        }
//                    }
//
//                    // matched text token, move text index to end of matched token
//                    textIdx += wcs.get(wcsIdx)!!.length
//                    anyChars = false
//                }
//                wcsIdx++
//            }
//
//            // full match
//            if (wcsIdx == wcs.size && textIdx == fileName.length) {
//                return true
//            }
//        } while (!backtrack.isEmpty())
//        return false
//    }

    /**
     * Splits a string into a number of tokens.
     * The text is split by '?' and '*'.
     * Where multiple '*' occur consecutively they are collapsed into a single '*'.
     *
     * @param text  the text to split
     * @return the array of tokens, never null
     */
    fun splitOnTokens(text: String): Array<String?> {
        // used by wildcardMatch
        // package level so a unit test may run on this
        if (text.indexOf('?') == NOT_FOUND && text.indexOf('*') == NOT_FOUND) {
            return arrayOf(text)
        }
        val array: CharArray = text.toCharArray()
        val list: ArrayList<String> = ArrayList()
        val buffer: StringBuilder = StringBuilder()
        var prevChar: Char = 0.toChar()
        for (ch: Char in array) {
            if (ch == '?' || ch == '*') {
                if (buffer.length != 0) {
                    list.add(buffer.toString())
                    buffer.setLength(0)
                }
                if (ch == '?') {
                    list.add("?")
                } else if (prevChar != '*') { // ch == '*' here; check if previous char was '*'
                    list.add("*")
                }
            } else {
                buffer.append(ch)
            }
            prevChar = ch
        }
        if (buffer.length != 0) {
            list.add(buffer.toString())
        }
        return list.toArray(EMPTY_STRING_ARRAY)
    }

    /**
     * Checks whether a given string is a valid host name according to
     * RFC 3986.
     *
     *
     * Accepted are IP addresses (v4 and v6) as well as what the
     * RFC calls a "reg-name". Percent encoded names don't seem to be
     * valid names in UNC paths.
     *
     * @see "https://tools.ietf.org/html/rfc3986.section-3.2.2"
     *
     * @param name the hostname to validate
     * @return true if the given name is a valid host name
     */
//    private fun isValidHostName(name: String): Boolean {
//        return isIPv6Address(name) || isRFC3986HostName(name)
//    }

    private val IPV4_PATTERN: Pattern = Pattern.compile("^(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$")
    private val IPV4_MAX_OCTET_VALUE: Int = 255

    /**
     * Checks whether a given string represents a valid IPv4 address.
     *
     * @param name the name to validate
     * @return true if the given name is a valid IPv4 address
     */
    // mostly copied from org.apache.commons.validator.routines.InetAddressValidator#isValidInet4Address
    private fun isIPv4Address(name: String): Boolean {
        val m: Matcher = IPV4_PATTERN.matcher(name)
        if (!m.matches() || m.groupCount() != 4) {
            return false
        }

        // verify that address subgroups are legal
        for (i in 1..4) {
            val ipSegment: String = m.group(i)
            val iIpSegment: Int = ipSegment.toInt()
            if (iIpSegment > IPV4_MAX_OCTET_VALUE) {
                return false
            }
            if (ipSegment.length > 1 && ipSegment.startsWith("0")) {
                return false
            }
        }
        return true
    }

    private val IPV6_MAX_HEX_GROUPS: Int = 8
    private val IPV6_MAX_HEX_DIGITS_PER_GROUP: Int = 4
    private val MAX_UNSIGNED_SHORT: Int = 0xffff
    private val BASE_16: Int = 16
    // copied from org.apache.commons.validator.routines.InetAddressValidator#isValidInet6Address
    /**
     * Checks whether a given string represents a valid IPv6 address.
     *
     * @param inet6Address the name to validate
     * @return true if the given name is a valid IPv6 address
     */
//    private fun isIPv6Address(inet6Address: String): Boolean {
//        val containsCompressedZeroes: Boolean = inet6Address.contains("::")
//        if (containsCompressedZeroes && (inet6Address.indexOf("::") != inet6Address.lastIndexOf("::"))) {
//            return false
//        }
//        if (((inet6Address.startsWith(":") && !inet6Address.startsWith("::"))
//                    || (inet6Address.endsWith(":") && !inet6Address.endsWith("::")))
//        ) {
//            return false
//        }
//        var octets: Array<String> = inet6Address.split(":".toRegex()).toTypedArray()
//        if (containsCompressedZeroes) {
//            val octetList: MutableList<String> = ArrayList(Arrays.asList(*octets))
//            if (inet6Address.endsWith("::")) {
//                // String.split() drops ending empty segments
//                octetList.add("")
//            } else if (inet6Address.startsWith("::") && !octetList.isEmpty()) {
//                octetList.removeAt(0)
//            }
//            octets = octetList.toArray(EMPTY_STRING_ARRAY)
//        }
//        if (octets.size > IPV6_MAX_HEX_GROUPS) {
//            return false
//        }
//        var validOctets: Int = 0
//        var emptyOctets: Int = 0 // consecutive empty chunks
//        for (index in octets.indices) {
//            val octet: String = octets.get(index)
//            if (octet.length == 0) {
//                emptyOctets++
//                if (emptyOctets > 1) {
//                    return false
//                }
//            } else {
//                emptyOctets = 0
//                // Is last chunk an IPv4 address?
//                if (index == octets.size - 1 && octet.contains(".")) {
//                    if (!isIPv4Address(octet)) {
//                        return false
//                    }
//                    validOctets += 2
//                    continue
//                }
//                if (octet.length > IPV6_MAX_HEX_DIGITS_PER_GROUP) {
//                    return false
//                }
//                var octetInt: Int = 0
//                try {
//                    octetInt = octet.toInt(BASE_16)
//                } catch (e: NumberFormatException) {
//                    return false
//                }
//                if (octetInt < 0 || octetInt > MAX_UNSIGNED_SHORT) {
//                    return false
//                }
//            }
//            validOctets++
//        }
//        return validOctets <= IPV6_MAX_HEX_GROUPS && (validOctets >= IPV6_MAX_HEX_GROUPS || containsCompressedZeroes)
//    }

    private val REG_NAME_PART_PATTERN: Pattern = Pattern.compile("^[a-zA-Z0-9][a-zA-Z0-9-]*$")

    /**
     * Checks whether a given string is a valid host name according to
     * RFC 3986 - not accepting IP addresses.
     *
     * @see "https://tools.ietf.org/html/rfc3986.section-3.2.2"
     *
     * @param name the hostname to validate
     * @return true if the given name is a valid host name
     */
    private fun isRFC3986HostName(name: String): Boolean {
        val parts: Array<String> = name.split("\\.".toRegex()).dropLastWhile({ it.isEmpty() }).toTypedArray()
        for (i in parts.indices) {
            if (parts.get(i).length == 0) {
                // trailing dot is legal, otherwise we've hit a .. sequence
                return i == parts.size - 1
            }
            if (!REG_NAME_PART_PATTERN.matcher(parts.get(i)).matches()) {
                return false
            }
        }
        return true
    }

    init {
        if (isSystemWindows) {
            OTHER_SEPARATOR = UNIX_SEPARATOR
        } else {
            OTHER_SEPARATOR = WINDOWS_SEPARATOR
        }
    }
}
