/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.application.properties

import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.PropertiesListener

open class WrappedMutableProperties<K>(properties: MutableProperties<K>) : MutableProperties<K> {
    private var properties: MutableProperties<K>?
    private val propertyMap: MutableMap<MutableProperty<Any>?, WrappedMutableProperty<*>?> = HashMap()
    private var listeners: MutableSet<PropertiesListener<K>>? = null
    override fun getProperty(name: K): MutableProperty<Any?> {
        val property = properties!!.getProperty(name)
        return getWrappedProperty<Any?>(property)
    }

    override fun getValue(name: K): Any? {
        return properties!!.getValue(name)
    }

    override fun addPropertiesListener(listener: PropertiesListener<K>) {
        getListeners()!!.add(listener)
        properties!!.addPropertiesListener(listener)
    }

    override fun addWeakPropertiesListener(listener: PropertiesListener<K>) {
        getListeners()!!.remove(listener)
        properties!!.addWeakPropertiesListener(listener!!)
    }

    override fun removePropertiesListener(listener: PropertiesListener<K>) {
        getListeners()!!.remove(listener)
        properties!!.removePropertiesListener(listener!!)
    }

    override operator fun iterator(): Iterator<K> {
        return properties!!.iterator()
    }

    override fun <T> createProperty(name: K, value: T): MutableProperty<T> {
        return getWrappedProperty<T>(properties!!.createProperty(name, value))
    }

    override fun <T> addProperty(name: K, property: MutableProperty<T>): MutableProperty<T> {
        return getWrappedProperty<T>(properties!!.addProperty(name, property))
    }

    override fun <T> replaceProperty(name: K, property: MutableProperty<T>): MutableProperty<T> {
        propertyMap!!.remove(properties!!.getProperty(name))
        return getWrappedProperty<T>(properties!!.replaceProperty(name, property))
    }

    override fun setValue(name: K, value: Any?) {
        properties!!.setValue(name, value)
    }

    protected fun <T> getWrappedProperty(property: MutableProperty<T>): MutableProperty<T> {
        if(propertyMap.containsKey(property)) {
            return propertyMap[property] as MutableProperty<T>
        } else {
            val wrappedProperty = WrappedMutableProperty(property)
            val casted = property as MutableProperty<Any>
            propertyMap[casted] = wrappedProperty
            return wrappedProperty
        }
    }

    private fun getListeners(): MutableSet<PropertiesListener<K>>? {
        if (listeners == null) {
            listeners = HashSet<PropertiesListener<K>>()
        }
        return listeners
    }

    open fun dispose() {
        for (wrappedMutableProperty in propertyMap.values) {
            wrappedMutableProperty!!.dispose()
        }
        if (listeners != null) {
            for (listener in listeners!!) {
                properties!!.removePropertiesListener(listener)
            }
            listeners = null
        }
        properties = null
    }

    init {
        this.properties = properties
    }
}