package com.macrofocus.application.document.swing

import com.macrofocus.application.document.*
import com.macrofocus.application.file.CPFileChooser
import com.macrofocus.application.file.CPFileManager
import com.macrofocus.application.file.DefaultFileManager
import com.macrofocus.application.properties.DocumentBasedApplicationProperties
import com.macrofocus.application.root.CPDragDropHere
import com.macrofocus.application.status.CPStatusBar
import com.macrofocus.application.window.CPWindowManager
import com.macrofocus.application.window.SwingMFIWindowManager
import com.macrofocus.common.command.Command
import com.macrofocus.common.command.Future
import com.macrofocus.common.command.FutureCommand
import com.macrofocus.common.command.UICommand
import com.macrofocus.common.file.CPFile
import com.macrofocus.common.logging.Logging
import com.macrofocus.common.properties.Property
import org.mkui.component.menu.CPMenu
import org.mkui.component.menu.CPMenuBar
import org.mkui.window.CPWindow
import java.awt.Desktop
import java.awt.Image
import java.awt.Taskbar
import java.io.File
import java.lang.Boolean
import java.lang.reflect.InvocationTargetException
import java.net.URL
import java.nio.file.Paths
import java.util.*
import java.util.concurrent.Callable
import java.util.concurrent.ExecutionException
import java.util.concurrent.FutureTask
import java.util.concurrent.RunnableFuture
import javax.swing.JFrame
import javax.swing.JMenuBar
import javax.swing.SwingUtilities
import kotlin.Array
import kotlin.Exception
import kotlin.ExperimentalStdlibApi
import kotlin.Nothing
import kotlin.RuntimeException
import kotlin.String
import kotlin.TODO
import kotlin.Throwable
import kotlin.Throws
import kotlin.UnsupportedOperationException
import kotlin.assert
import kotlin.let


/**
 * Created by luc on 02/12/15.
 */
@ExperimentalStdlibApi
abstract class SwingSDIDocumentBasedApplication<D : Document, V : View<D>> : AbstractDocumentBasedApplication<D, V>() {
    @ExperimentalStdlibApi
    private var state: DocumentBasedApplicationState<D, V>? = null
    var windowManager: CPWindowManager<V>? = null
        private set
    var fileManager: CPFileManager? = null
        private set
    protected var allowNewDocument = true
    protected var startWithNewDocument = true
    protected var exitOnClose = false
    protected var openFileAutomatic = true
    private var initialized = false
    private var started = false
    private val startDocuments: MutableList<DocumentOpener<D>> =
        ArrayList<DocumentOpener<D>>()
    var fileChooser: CPFileChooser? = null
        get() {
            if (field == null) {
                field = CPFileChooser()
            }
            return field
        }
        private set
    private val listener: CPDragDropHere.Listener = object : CPDragDropHere.Listener {
        override fun filesDropped(files: Iterable<File?>?) {
                    for (file in files!!) {
                        val cpFile = CPFile(file!!)
                        val documentOpener: DocumentOpener<D> = object : DocumentOpener<D> {
                            @Throws(OpenDocumentException::class)
                            override fun openDocument(window: CPWindow?): D? {
                                return openLocalDocument(cpFile, null, window)
                            }
                        }
                        val newWindow: CPWindow = createWindow()
                        openAndShowDocument(newWindow, documentOpener, cpFile)
                    }
        }
    }
    val recent: Array<String?>?
        get() = null
    @ExperimentalStdlibApi
    override val applicationState: DocumentBasedApplicationState<D, V>?
        get() = state

    override fun init() {
        assert(!initialized)

//        RepaintManager.setCurrentManager(new CheckThreadViolationRepaintManager());
        windowManager = SwingMFIWindowManager(this)
        fileManager = DefaultFileManager()
        System.setProperty("apple.laf.useScreenMenuBar", "true")
        System.setProperty("apple.awt.application.name", name)
        System.setProperty("com.apple.mrj.application.apple.menu.about.name", name)
        System.setProperty("http.agent", name)
        state = DocumentBasedApplicationState(recent)
        val desktop: CPDesktop? = CPDesktop.instance
        if (desktop != null) {
            try {
                if (desktop.isSupported(Desktop.Action.APP_OPEN_FILE)) {
                    desktop.setOpenFileHandler { e ->
                        for (file in e.getFiles()) {
                            val cpFile = CPFile(file)
                            try {
                                val documentOpener: LocalDocumentOpener<D> =
                                    object : LocalDocumentOpener<D> {
                                        override val file: CPFile
                                            get() {
                                                return cpFile
                                            }

                                        @Throws(OpenDocumentException::class)
                                        override fun openDocument(window: CPWindow?): D? {
                                            return openLocalDocument(cpFile, null as String, window)
                                        }
                                    }
                                if (!initialized) {
                                    startDocuments.add(documentOpener)
                                } else {
                                    val newWindow: CPWindow = createWindow()
                                    openAndShowDocument(newWindow, documentOpener, cpFile)
                                }
                            } catch (e1: Throwable) {
                                Logging.instance.process(e1)
                            }
                        }
                    }
                }
                if (desktop.isSupported(Desktop.Action.APP_OPEN_URI)) {
                    desktop.setOpenURIHandler { e ->
                        val window: CPWindow = createWindow()
                        try {
                            val url: URL = e.getURI().toURL()
                            if (!initialized) {
                                startDocuments.add(object : DocumentOpener<D> {
                                    @Throws(OpenDocumentException::class)
                                    override fun openDocument(window: CPWindow?): D? {
                                        return openRemoteDocument(url.file, url.toExternalForm(), null)
                                    }
                                })
                            } else {
                                setup(
                                    window,
                                    createView(
                                        window,
                                        openRemoteDocument(url.file, url.toExternalForm(), null)
                                    )
                                )
                            }
                        } catch (e1: Throwable) {
                            Logging.instance.process(e1)
                        }
                    }
                }
                if (desktop.isSupported(Desktop.Action.APP_ABOUT)) {
                    desktop.setAboutHandler { e -> showAbout(windowManager!!.activeWindow) }
                }
                if (desktop.isSupported(Desktop.Action.APP_QUIT_HANDLER)) {
                    desktop.setQuitHandler { e, response -> stop() }
                }
            } catch (e: NoSuchFieldError) {
                // Java 8 or lower
            }
        }
        try {
            if (Taskbar.isTaskbarSupported()) {
                try {
                    if (applicationIcon != null) {
                        Taskbar.getTaskbar().iconImage = applicationIcon
                    }
                } catch (e: SecurityException) {
                    Logging.instance.process(e)
                } catch (e: UnsupportedOperationException) {
                }
            }
        } catch (e: NoClassDefFoundError) {
            // Java 8 or lower
            e.printStackTrace()
        }
        //        if(SystemTray.isSupported()) {
//            try {
//                SystemTray.getSystemTray().add(new TrayIcon(getApplicationIcon(), getName()));
//            } catch (AWTException e) {
//                e.printStackTrace();
//            }
//        }
        initialized = true
    }

    open fun showAbout(window: CPWindow?) {}
    fun start(fileUrl: String?) {
        assert(initialized)
        assert(!started)
        val windowExist = windowManager!!.windowCount == 0
        val window: CPWindow? = windowManager!!.createWindow()
        val firstWindow: CPWindow? = window
        if (fileUrl != null) {
            try {
                if (fileUrl.startsWith("http")) {
                    startDocuments.add(object : DocumentOpener<D> {
                        @Throws(OpenDocumentException::class)
                        override fun openDocument(window: CPWindow?): D? {
                            return openRemoteDocument(fileUrl, fileUrl, null)
                        }
                    })
                } else {
                    val file = CPFile(File(fileUrl))
                    startDocuments.add(object : LocalDocumentOpener<D> {
                        override val file: CPFile
                            get() = file

                        @Throws(OpenDocumentException::class)
                        override fun openDocument(window: CPWindow?): D? {
                            return openLocalDocument(file, null, window)
                        }
                    })
                }
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }
        if (startDocuments.isEmpty() && openFileAutomatic) {
            try {
                // Only open file automatically if no other window is open (e.g. a document may have been opened directly)
                if (windowExist) {
                    for (directory in installLocations) {
                        // Supposedly smarter code that doesn't even work
                        //                    try {
                        //                        Files.find(directory.toPath(), 1, new BiPredicate<Path,BasicFileAttributes>() {
                        //                            @Override
                        //                            public boolean test(Path path, BasicFileAttributes basicFileAttributes) {
                        //                                return basicFileAttributes.isRegularFile()
                        //                                        && path.getFileName().toString().matches(".*\\." + getOpenFileTypes()[0].getExtensions​()[0]);
                        //                            }
                        //                        }).findFirst().ifPresent(new Consumer<Path>() {
                        //                            @Override
                        //                            public void accept(Path path) {
                        //                                System.err.println("Launching " + path);
                        ////                                showDocument(window, openLocalDocument(path.toFile(), null));
                        //                                final D document = openLocalDocument(path.toFile(), null);
                        //                                System.err.println(document.getName());
                        //                                setup(window, createView(window, document));
                        //                            }
                        //                        });
                        //                    } catch (IOException e) {
                        //                        e.printStackTrace();
                        //                    }
                        val files = directory.listFiles { dir, name ->
                            val file = openFileTypes.get(0)
                            name.endsWith(
                                "." + file.extensions.get(0)
                            )
                        }
                        if (files != null && files.size == 1) {
                            val file = CPFile(files[0])
                            startDocuments.add(object : LocalDocumentOpener<D> {
                                override val file: CPFile
                                    get() = file

                                @Throws(OpenDocumentException::class)
                                override fun openDocument(window: CPWindow?): D? {
                                    return openLocalDocument(file, null, window)
                                }
                            })
                            break
                        }
                    }
                }
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }
        if (!BACKGROUND_LOADING) {
            var success = false
            if (startDocuments.isEmpty()) {
                start(window!!)
                success = true
            } else {
                var currentWindow: CPWindow? = window
                for (documentOpener in startDocuments) {
                    val newWindow: CPWindow?
                    newWindow = if (currentWindow != null) {
                        currentWindow
                    } else {
                        windowManager!!.createWindow()
                    }
                    try {
                        val document: D? = documentOpener.openDocument(newWindow)
                        if (documentOpener is LocalDocumentOpener) {
                            fileManager!!.setFile(
                                newWindow,
                                (documentOpener as LocalDocumentOpener).file
                            )
                        }
                        setup(newWindow, createView(newWindow, document))
                        if (currentWindow != null) {
                            currentWindow = null
                        }
                        success = true
                    } catch (e: Throwable) {
                        Logging.instance.process(e)
                        if (newWindow !== currentWindow) {
                            windowManager!!.closeWindow(newWindow)
                        }
                    }
                }
            }

            // Only open a new blank window if no other window is open (e.g. a document may have been opened directly)
            if (!success) {
                start(window!!)
            }
            started = true
        } else {
            runAsyncBackground(object : Command {
                var w: CPWindow? = window
                override fun execute() {
                    var success = false
                    if (startDocuments.isEmpty()) {
                        try {
                            success = invokeAndWait {
                                start(w!!)
                                Boolean.TRUE
                            }
                        } catch (e: Exception) {
                            Logging.instance.process(e)
                        }
                    } else {
                        for (documentOpener in startDocuments) {
                            var newWindow: CPWindow? = null
                            if (w != null) {
                                newWindow = w
                            } else {
                                try {
                                    newWindow = invokeAndWait(Callable<CPWindow?> { windowManager!!.createWindow() })
                                } catch (e: Exception) {
                                    Logging.instance.process(e)
                                }
                            }
                            try {
                                val document: D? = documentOpener.openDocument(newWindow)
                                val nw: CPWindow? = newWindow
                                invokeAndWait<Nothing?>(Callable {
                                    if (documentOpener is LocalDocumentOpener) {
                                        fileManager!!.setFile(
                                            nw,
                                            (documentOpener as LocalDocumentOpener).file
                                        )
                                    }
                                    setup(nw, createView(nw, document))
                                    null
                                })
                                if (w != null) {
                                    w = null
                                }
                                success = true
                            } catch (e: Throwable) {
                                Logging.instance.process(e)
                                if (newWindow !== w) {
                                    windowManager!!.closeWindow(newWindow!!)
                                }
                            }
                        }
                    }

                    // Only open a new blank window if no other window is open (e.g. a document may have been opened directly)
                    if (!success) {
                        start(w!!)
                    }
                    started = true
                }
            })
        }
    }

    // Find default settings
    private val installLocations: Iterable<File>
        private get() {
            // Find default settings
            val directories: MutableSet<File> = LinkedHashSet()
            val currentRelativePath = Paths.get("")
            directories.add(currentRelativePath.toFile())
            val userDir = System.getProperty("user.dir")
            if (userDir != null) {
                val dir = File(userDir)
                if (dir.exists()) {
                    directories.add(dir)
                }
            }
            val appDir = System.getProperty("install4j.appDir")
            if (appDir != null) {
                val dir = File(appDir)
                if (dir.exists()) {
                    directories.add(dir)
                }
            }
            val exeDir = System.getProperty("install4j.exeDir")
            if (exeDir != null) {
                val dir = File(exeDir)
                if (dir.exists()) {
                    directories.add(dir)
                }
            }
            return directories
        }

    protected fun start(firstWindow: CPWindow) {
        if (startWithNewDocument) {
            setup(firstWindow, createView(firstWindow, newDocument()))
        } else {
            setup(firstWindow, createView(firstWindow, null))
        }
    }

    override fun stop() {
        applicationState!!.triggerClossing()
        System.exit(0)
    }

    override fun createWindow(): CPWindow {
        return windowManager!!.createWindow()
    }

    override fun showDocument(window: CPWindow?, document: D?) {
        val newWindow: CPWindow = createWindow()
        val view: V? = createView(newWindow, document)
        setup(newWindow, view)
    }

    override fun openAndShowDocument(window: CPWindow?, documentOpener: DocumentOpener<D>, file: CPFile?) {
        val newWindow: CPWindow = createWindow()
        if (BACKGROUND_LOADING) {
            runAsyncBackground(object : Command {
                override fun execute() {
                    try {
                        val document: D? = documentOpener.openDocument(newWindow)
                        SwingUtilities.invokeAndWait {
                            if (file != null) {
                                fileManager!!.setFile(newWindow, file)
                            }
                            setup(newWindow, createView(newWindow, document))
                        }
                    } catch (t: Throwable) {
                        Logging.instance.process(t)
                        try {
                            windowManager!!.closeWindow(newWindow)
                        } catch (e: Exception) {
                            Logging.instance.process(e)
                        }
                    }
                }
            })
        } else {
            try {
                val document: D? = documentOpener.openDocument(newWindow)
                if (file != null) {
                    fileManager!!.setFile(newWindow, file)
                }
                setup(window, createView(newWindow, document))
            } catch (t: Throwable) {
                Logging.instance.process(t)
                windowManager!!.closeWindow(newWindow)
            }
        }
    }

    fun configureWindow(window: JFrame) {
        window.iconImage = applicationIcon
    }

    @ExperimentalStdlibApi
    fun closeWindow(window: CPWindow?) {
        val view: V? = applicationState!!.getView(window)
        view?.let { closeView(it) }
    }

    protected open val applicationIcon: Image?
        protected get() = null

    override fun setup(window: CPWindow?, view: V?) {
        state!!.remove(window)
        state!!.remove(view)
        state!!.add(window, view)
        var title: String = name!!
        if (view!!.document != null) {
            title += " - " + view.document!!.documentState!!.name
        }
        window!!.nativeWindow.setTitle(title)
        view.customizeWindow(window)
        val menuBar = createMenuBar(window, view)
        val statusBar: CPStatusBar = createStatusBar(window, view)
        window.setMenuBar(menuBar)
        window.setStatusBar(statusBar)
        if (view.document != null) {
            window.content = view.nativeComponent
            val dragDropHere = CPDragDropHere(window.getNativeComponent())
            dragDropHere.addListener(listener)
        } else {
            val dragDropHere = CPDragDropHere(window.getNativeComponent())
            customizeEmptyView(dragDropHere, window)
            dragDropHere.addListener(listener)
            window.content = dragDropHere
        }
        windowManager!!.assign(window.component, window, view.document == null)
    }

    protected open fun customizeEmptyView(dragDropHere: CPDragDropHere?, window: CPWindow?) {}
    protected fun createMenuBar(window: CPWindow, view: V): JMenuBar {
        val cpMenuBar = CPMenuBar()
        val menuBar: JMenuBar = cpMenuBar.nativeComponent
        val fileMenu: CPMenu =
            cpMenuBar.addMenu(view.properties.getProperty(DocumentBasedApplicationProperties.MENU_FILE_LABEL) as Property<String?>)
        if (allowNewDocument) {
            fileMenu.addAction(NewUICommand(this as DocumentBasedApplication<D, View<D>>, view))
        }
        fileMenu.addEllipsisAction(OpenUICommand(this as DocumentBasedApplication<D, View<D>>, view, window))
        fileMenu.addEllipsisAction(OpenURLUICommand(this, view))
        view.customizeFileOpen(fileMenu)
        val reopenMenu: CPMenu =
            fileMenu.addMenu(view.properties.getProperty(DocumentBasedApplicationProperties.MENU_OPENRECENT_LABEL) as Property<String?>)
        restoreReopen(reopenMenu, view)
        val reloadCommand: UICommand? = getReloadCommand(view)
        if (reloadCommand != null) {
            fileMenu.addAction(reloadCommand)
        }
        fileMenu.addSeparator()
        if (saveFileTypes != null) {
            fileMenu.addSeparator()
            val saveAsUICommand: SaveAsUICommand<D> = SaveAsUICommand(this, view, window)
            fileMenu.addAction(SaveUICommand<D>(this, view, window, saveAsUICommand))
            fileMenu.addEllipsisAction(saveAsUICommand)
        }
        view.customizeFileSave(fileMenu)
        fileMenu.addSeparator()
        fileMenu.addAction(CloseUICommand(this, view))
        fileMenu.addAction(ExitUICommand(this, view))
        view.customizeMenuBar(cpMenuBar)
        return menuBar
    }

    /* ToDo: memory leak! */
    private fun restoreReopen(reopenMenu: CPMenu, view: V) {
//        getApplicationState().addListener(new DocumentBasedApplicationState.Listener() {
//            @Override
//            public void recentChanged() {
//                populateReopen(reopenMenu);
//            }
//        });
        populateReopen(reopenMenu, view)
    }

    private fun populateReopen(reopenMenu: CPMenu, view: V) {
        reopenMenu.removeAll()
        for (documentState in applicationState!!.getRecent()!!) {
            reopenMenu.addAction(OpenRecentUICommand(this as DocumentBasedApplication<D, View<D>>, view, documentState!!))
        }
    }

    private fun saveReopen(reopenMenu: CPMenu) {}
    protected fun createStatusBar(window: CPWindow?, view: V): CPStatusBar {
        val statusBar = CPStatusBar()
        view.customizeStatusBar(statusBar)
        return statusBar
    }

    override fun runAsyncUIThread(command: Command) {
        SwingUtilities.invokeLater {
            try {
                command.execute()
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }
    }

    override fun runAsyncBackground(command: Command) {
        Thread {
            try {
                command.execute()
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }.start()
    }

    override fun <C : Command?, R> runAsyncBackground(
        command: C?,
        callback: DocumentBasedApplication.Callback<C?, R?>?
    ) {
        Thread {
            try {
                command!!.execute()
                callback!!.execute(command)
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }.start()
    }

    override fun <C : FutureCommand<R?>?, R> runAsyncFuture(command: FutureCommand<R?>?): Future<R?>? {
        val callable = Callable { command!!.execute() }
        val future: FutureTask<R?> = FutureTask<R?>(callable)
        Thread(future).start()
        //        ExecutorService executor = Executors.newFixedThreadPool(1);
//        java.util.concurrent.Future<R> future = executor.submit(callable);
        return object : Future<R?> {
            override val isCancelled: kotlin.Boolean
                get() = false

            override val isDone: kotlin.Boolean
                get() = TODO("Not yet implemented")

            @Throws(Exception::class)
            override fun get(): R? {
                return future.get()
            }

            override fun cancel(mayInterruptIfRunning: kotlin.Boolean): kotlin.Boolean {
                TODO("Not yet implemented")
            }
        }
    }

    override fun closeView(view: V?) {
        super.closeView(view)
        val window: CPWindow? = applicationState!!.getWindow(view)
        applicationState!!.remove(window)
        applicationState!!.remove(view)
        fileManager!!.removeWindow(window)
        if (!applicationState!!.isEmpty) {
            if (window != null) {
                // Close the view
                windowManager!!.closeWindow(window)
            } else {
                stop()
            }
        } else {
            if (exitOnClose || view!!.document == null) {
                // Exit
                windowManager!!.closeWindow(window!!)
                stop()
            } else {
                // Reuse the window to open a blank view
                setup(window, createView(window, null))
            }
        }
    }

    fun openFileChooser(window: CPWindow) {
        val fileChooser: CPFileChooser = fileChooser!!
        fileChooser.setTitle("Open File")
        fileChooser.setFileTypes(*openFileTypes)
        val file: CPFile? = fileChooser.showOpenDialog(window)
        if (file != null) {
            try {
                val documentOpener: DocumentOpener<D> =
                    object : DocumentOpener<D> {
                        @Throws(OpenDocumentException::class)
                        override fun openDocument(window: CPWindow?): D? {
                            return openLocalDocument(file, null, window)
                        }
                    }
                openAndShowDocument(window, documentOpener, file)
            } catch (e: Throwable) {
                Logging.instance.process(e)
            }
        }
    }

    companion object {
        private const val BACKGROUND_LOADING = true
        fun <T> invokeLater(callable: Callable<T>?): FutureTask<T> {
            val task = FutureTask(callable)
            SwingUtilities.invokeLater(task)
            return task
        }

        @Throws(InterruptedException::class, InvocationTargetException::class)
        fun <T> invokeAndWait(callable: Callable<T>): T {
            return if (SwingUtilities.isEventDispatchThread()) {
                try {
                    //blocks until future returns
                    callable.call()
                } catch (e: Exception) {
                    val t = e.cause
                    if (t is RuntimeException) {
                        throw (t as RuntimeException?)!!
                    } else if (t is InvocationTargetException) {
                        throw (t as InvocationTargetException?)!!
                    } else {
                        throw InvocationTargetException(t)
                    }
                }
            } else {
                try {
                    //blocks until future returns
                    invokeLater(callable).get()
                } catch (e: ExecutionException) {
                    val t = e.cause
                    if (t is RuntimeException) {
                        throw (t as RuntimeException?)!!
                    } else if (t is InvocationTargetException) {
                        throw (t as InvocationTargetException?)!!
                    } else {
                        throw InvocationTargetException(t)
                    }
                }
            }
        }

        @Throws(InterruptedException::class, ExecutionException::class)
        fun <V> getFromEDT(callable: Callable<V>?): V {
            val f: RunnableFuture<V> = FutureTask(callable)
            if (SwingUtilities.isEventDispatchThread()) {
                f.run()
            } else {
                SwingUtilities.invokeLater(f)
            }
            return f.get()
        }
    }
}
