package com.macrofocus.application.window

import com.macrofocus.application.document.View
import com.macrofocus.application.document.swing.SwingSDIDocumentBasedApplication
import org.mkui.component.CPComponent
import org.mkui.window.CPWindow
import java.awt.Frame
import java.awt.Toolkit
import java.awt.event.WindowAdapter
import java.awt.event.WindowEvent
import javax.swing.FocusManager
import javax.swing.JFrame

/**
 * MFI (multiple frame interface). MFIs place each document into a completely separate window, which gets a separate button on the taskbar. When you open multiple documents in an MFI application, it looks as though there are multiple instances of the application running at once (similar to an SDI application). However, the underlying architecture is different. Word is an MFI application—even though each document has its own separate window, you can use the Window menu to jump from one to another, because they’re really all part of one application.
 *
 * @param <V>
</V> */
@ExperimentalStdlibApi
class SwingMFIWindowManager<V : View<*>?>(private val application: SwingSDIDocumentBasedApplication<*, *>) : AbstractWindowManager<V>() {
    private val singleNewWindow = true
    private val windows: MutableList<CPWindow>
    override var newWindow: CPWindow? = null
    override val windowCount: Int
        get() = windows.size

    override fun createWindow(): CPWindow {
        return if (singleNewWindow && newWindow != null) {
            val window: CPWindow = newWindow!!
            if (Toolkit.getDefaultToolkit().isFrameStateSupported(Frame.ICONIFIED)) {
                if (isFrameMinimized(window)) {
                    window.nativeWindow.setState(Frame.NORMAL)
                }
            }
            window.nativeWindow.toFront()
            window.nativeWindow.requestFocus()
            window
        } else {
            val window: CPWindow
            window = CPWindow()
            configureWindow(window)
            window.nativeWindow.setSize(1000, 720)
            window.nativeWindow.setLocationRelativeTo(null)
            //            if (!windows.isEmpty()) {
//            window.nativeWindow.setLocationByPlatform(true)
            //                window.setLocation(window.getX() + 32, window.getY() + 32);
            //            }
            window.nativeWindow.setVisible(true)
            window.nativeWindow.toFront()
            window.nativeWindow.requestFocus()
            windows.add(window)
            newWindow = window
            window
        }
    }

    fun isFrameMinimized(frame: CPWindow): Boolean {
        return frame.nativeWindow.getExtendedState() and Frame.ICONIFIED != 0
    }

    @ExperimentalStdlibApi
    protected fun configureWindow(window: CPWindow) {
        window.nativeWindow.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE)
        window.nativeWindow.addWindowListener(object : WindowAdapter() {
            override fun windowClosing(e: WindowEvent) {
                application.closeWindow(window)
            }
        })
        application.configureWindow(window.nativeWindow)
    }

    override fun closeWindow(window: CPWindow) {
        window.nativeWindow.getContentPane().removeAll()
        if (newWindow === window) {
            newWindow = null
        }
        windows.remove(window)
        window.nativeWindow.dispose()
    }

    override fun assign(component: CPComponent?, window: CPWindow, newView: Boolean) {
        if (newView) {
            newWindow = window
        } else {
            if (newWindow === window) {
                newWindow = null
            }
        }
    }

    override val activeWindow: CPWindow?
        get() {
            val activeWindow = FocusManager.getCurrentManager().activeWindow
            if (activeWindow != null) {
                for (window in windows) {
                    if (activeWindow === window.nativeWindow) {
                        return window
                    }
                }
            }
            return null
        }

    init {
        windows = ArrayList<CPWindow>()
        newWindow = null
    }
}
