/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

/**
 * This interface represents the current state of a selection. A selection is to be understood in the broad sense and
 * not only related to user interactions.
 *
 * @param <E> the type of elements that can be selected
</E> */
interface Selection<E> : Iterable<E> {
    /**
     * Indicates whether at least one element is currently selected.
     *
     * @return true if at least one element is selected, false otherwise.
     */
    val isActive: Boolean

    /**
     * Indicates whether a given element is selected or not.
     *
     * @param element the element to be tested
     * @return true if the element is selected, false otherwise
     */
    fun isSelected(element: E?): Boolean

    /**
     * Returns the selected elements as a Set.
     *
     * @return the Set of selected elements.
     */
    val selectedSet: Set<E>

    /**
     * Returns the number of elements currently selected.
     *
     * @return the number of selected elements.
     */
    val selectedCount: Int

    /**
     * Add a listener to the list that's notified each time a change to the selection occurs.
     *
     * @param listener the SelectionListener
     */
    fun addSelectionListener(listener: SelectionListener<E>)

    /**
     * Add a listener to the list that's notified each time a change to the selection occurs. The listener will
     * automatically be disposed of should no other object have a reference to it.
     *
     * @param listener the SelectionListener
     */
    fun addWeakSelectionListener(listener: SelectionListener<E>)

    /**
     * Remove a listener to the list that's notified each time a change to the selection occurs.
     *
     * @param listener the SelectionListener
     */
    fun removeSelectionListener(listener: SelectionListener<E>)

    /**
     * Remove all listeners to the list that's notified each time a change to the selection occurs.
     */
    fun removeSelectionListeners()
    val isEnabled: Boolean
}