/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

import com.macrofocus.common.collection.CollectionFactory

/**
 * Default data model for multiple selection. Selecting a null value is treated as clearing the selection.
 *
 * @param <E> the type of elements that can be selected
</E> */
class SimpleSelection<E>() : AbstractMutableSelection<E>(), MutableSelection<E> {
    private val selected: MutableSet<E>

    constructor(vararg selected: E) : this() {
        this.selected.addAll(selected)
    }

    override fun clearSelection(): Boolean {
        return if (isEnabled && isActive) {
            val changes: MutableSet<E> = HashSet()
            changes.addAll(selected)
            selected.clear()
            notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
            true
        } else {
            false
        }
    }

    override val isActive: Boolean
        get() = !selected.isEmpty() && isEnabled

    override fun isSelected(element: E?): Boolean {
        return element != null && selected.contains(element)
    }

    override fun setSelectedElement(e: E) {
        if (isEnabled) {
            if (e != null) {
                val changes: MutableSet<E> = HashSet()
                for (element in selected) {
                    changes.add(element)
                }
                if (!isSelected(e)) {
                    changes.add(e)
                    selected.add(e)
                } else {
                    changes.remove(e)
                }
                if (changes.size > 0) {
                    selected.clear()
                    selected.add(e)
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                }
            } else {
                clearSelection()
            }
        }
    }

    override fun setSelectedState(element: E, selected: Boolean) {
        if (isEnabled && element != null) {
            if (selected) {
                if (!isSelected(element)) {
                    val changes: MutableSet<E> = HashSet()
                    changes.add(element)
                    this.selected.add(element)
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                }
            } else {
                if (isSelected(element)) {
                    val changes: MutableSet<E> = HashSet()
                    changes.add(element)
                    this.selected.remove(element)
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                }
            }
        }
    }

    override fun setSelectedIterableState(elements: Iterable<E>, selected: Boolean): Boolean {
        if (isEnabled && elements != null) {
            if (selected) {
                val changes: MutableSet<E> = HashSet()
                for (element in elements) {
                    if (!isSelected(element)) {
                        changes.add(element)
                        this.selected.add(element)
                    }
                }
                if (changes.size > 0) {
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                    return true
                }
            } else {
                val changes: MutableSet<E> = HashSet()
                for (element in elements) {
                    if (isSelected(element)) {
                        changes.add(element)
                        this.selected.remove(element)
                    }
                }
                if (changes.size > 0) {
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                    return true
                }
            }
        }
        return false
    }

    override fun setSelectedElementsState(selected: Boolean, vararg elements: E): Boolean {
        if (isEnabled && elements != null) {
            if (selected) {
                val changes: MutableSet<E> = HashSet()
                for (element in elements) {
                    if (!isSelected(element)) {
                        changes.add(element)
                        this.selected.add(element)
                    }
                }
                if (changes.size > 0) {
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                    return true
                }
            } else {
                val changes: MutableSet<E> = HashSet()
                for (element in elements) {
                    if (isSelected(element)) {
                        changes.add(element)
                        this.selected.remove(element)
                    }
                }
                if (changes.size > 0) {
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                    return true
                }
            }
        }
        return false
    }

    override fun setSelectedElements(vararg elements: E) {
        if (isEnabled && elements != null) {
            val changes: MutableSet<E> = HashSet()
            for (element in selected) {
                changes.add(element)
            }
            for (element in elements) {
                if (!isSelected(element)) {
                    changes.add(element)
                    selected.add(element)
                } else {
                    changes.remove(element)
                }
            }
            if (changes.size > 0) {
                selected.clear()
                selected.addAll(elements)
                notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
            }
        }
    }

    override fun setSelectedIterable(elements: Iterable<E>) {
        if (isEnabled) {
            if (elements != null) {
                val changes: MutableSet<E> = HashSet()
                for (element in selected) {
                    changes.add(element)
                }
                for (element in elements) {
                    if (!isSelected(element)) {
                        changes.add(element)
                        selected.add(element)
                    } else {
                        changes.remove(element)
                    }
                }
                if (changes.size > 0) {
                    selected.clear()
                    for (element in elements) {
                        selected.add(element)
                    }
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                }
            } else {
                if (!selected.isEmpty()) {
                    val changes: MutableSet<E> = HashSet()
                    for (element in selected) {
                        changes.add(element)
                    }
                    selected.clear()
                    notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
                }
            }
        }
    }

    override val selectedCount: Int
        get() = selected.size

    override val selectedSet: Set<E>
        get() = selected

    override operator fun iterator(): Iterator<E> {
        return selected.iterator()
    }

    override fun toString(): String {
        var selected = ""
        var first = true
        for (s in this.selected) {
            if (!first) {
                selected += ","
            } else {
                first = false
            }
            selected += s.toString()
        }
        return "SimpleSelection{" +
                "selected=" + selected +
                '}'
    }

    init {
        selected = CollectionFactory.concurrentLinkedHashSet<E>()
    }
}