/*
 * Copyright (c) 2006, Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.hierarchy

import kotlin.math.max
import kotlin.math.min

/**
 * A simple mutable hierarchy implementation.
 */
class SimpleHierarchy<T>(override val root: T) : AbstractHierarchy<T>(), MutableHierarchy<T> {
    private val parentChildren: MutableMap<T, MutableList<T>>
    private val childParent: MutableMap<T, T>

    constructor(hierarchy: Hierarchy<T>) : this(hierarchy.root) {
        copyFrom(hierarchy)
    }

    override fun getParent(child: T): T? {
        return childParent[child]
    }

    override fun hasChild(parent: T): Boolean {
        return parentChildren.containsKey(parent)
    }

    override fun getChildren(parent: T): Iterable<T>? {
        return getChildList(parent)
    }

    override fun getChild(parent: T, index: Int): T {
        return getChildList(parent)!![index]
    }

    override fun getChildCount(parent: T): Int {
        return getChildList(parent)!!.size
    }

    override fun getIndexOfChild(parent: T, child: T): Int {
        return getChildList(parent)!!.indexOf(child)
    }

    override fun containsChild(parent: T, child: T): Boolean {
        return getChildList(parent)!!.contains(child)
    }

    override fun containsChild(child: T): Boolean {
        return childParent.containsKey(child)
    }

    override fun getChildList(parent: T): List<T>? {
        return if (parentChildren.containsKey(parent)) {
            parentChildren[parent]
        } else {
            emptyList()
        }
    }

    private fun addChild(parent: T, child: T, isAdjusting: Boolean): T {
//        assert !containsChild(child) : "Child already exist " + child;
        return if (!parentChildren.containsKey(child)) {
            val list: MutableList<T>
            if (!parentChildren.containsKey(parent)) {
                list = ArrayList<T>()
                parentChildren[parent] = list
            } else {
                list = parentChildren[parent]!!
            }
            list.add(child)
            childParent[child] = parent

//            final int index = getIndexOfChild(parent, child);
            val index = list.size - 1
            notifyHierarchyNodeInserted(child, parent, index, isAdjusting)
            child
        } else {
            throw IllegalArgumentException("Child $child already exists")
        }
    }

    override fun addChild(parent: T, child: T): T {
        return addChild(parent, child, false)
    }

    override fun setChildren(parent: T, children: List<T>) {
        val toRemove: MutableList<T> = ArrayList<T>()
        for (child in getChildren(parent)!!) {
            toRemove.add(child)
        }
        for (i in toRemove.indices) {
            val child = toRemove[i]
            removeChild(parent, child, children.size > 0 || i < toRemove.size - 1)
        }
        for (i in children.indices) {
            val child = children[i]
            addChild(parent, child, i < children.size - 1)
        }
    }

    override fun addChildren(parent: T, children: List<T>) {
        for (i in children.indices) {
            val child = children[i]
            addChild(parent, child, i < children.size - 1)
        }
    }

    override fun removeChildren(children: List<T>) {
        for (i in children.indices) {
            val child = children[i]
            removeChild(getParent(child)!!, child, i < children.size - 1)
        }
    }

    override fun insertChild(parent: T, index: Int, child: T) {
        if (!parentChildren.containsKey(child)) {
            val list: MutableList<T>
            if (!parentChildren.containsKey(parent)) {
                list = ArrayList<T>()
                parentChildren[parent] = list
            } else {
                list = parentChildren[parent]!!
            }
            list.add(index, child)
            childParent[child] = parent

//            final int index = getIndexOfChild(parent, child);
            notifyHierarchyNodeInserted(child, parent, index, false)
        } else {
            throw IllegalArgumentException("Child $child already exists")
        }
    }

    override fun removeAll() {
        parentChildren.clear()
        childParent.clear()
        notifyHierarchyStructureChanged()
    }

    override fun removeChild(parent: T, child: T) {
        removeChild(parent, child, false)
    }

    private fun removeChild(parent: T, child: T, isAdjusting: Boolean) {
        if (!parentChildren.containsKey(child)) {
            if (childParent.containsKey(child)) {
                childParent.remove(child)
            } else {
//                throw new IllegalArgumentException("Child " + child + " cannot be found");
            }
            val index = getIndexOfChild(parent, child)
            val list = parentChildren[parent]!!
            list.remove(child)
            if (list.isEmpty()) {
                parentChildren.remove(parent)
            }
            notifyHierarchyNodeRemoved(child, parent, index, isAdjusting)
        } else {
            throw IllegalArgumentException("Child $child contains children")
        }
    }

    override fun moveChild(oldParent: T, newParent: T, insertionPoint: Int, child: T) {
        @Suppress("NAME_SHADOWING") var insertionPoint = insertionPoint
        run {
            if (childParent.containsKey(child)) {
                childParent.remove(child)
            } else {
                //                throw new IllegalArgumentException("Child " + child + " cannot be found");
            }
            val index = getIndexOfChild(oldParent, child)
            val list = parentChildren[oldParent]!!
            list.remove(child)
            if (list.isEmpty()) {
                parentChildren.remove(oldParent)
            }
            notifyHierarchyNodeRemoved(child, oldParent, index, true)
        }
        val list: MutableList<T>
        if (!parentChildren.containsKey(newParent)) {
            list = ArrayList<T>()
            parentChildren[newParent] = list
        } else {
            list = parentChildren[newParent]!!
        }
        insertionPoint = min(insertionPoint, list.size - 1)
        insertionPoint = max(0, insertionPoint)
        list.add(insertionPoint, child)
        childParent[child] = newParent
        notifyHierarchyNodeInserted(child, newParent, insertionPoint, false)
    }

    private fun copyFrom(source: Hierarchy<T>) {
        recursiveCopyFrom(source, source.root)
    }

    private fun recursiveCopyFrom(source: Hierarchy<T>, parent: T) {
        for (child in source.getChildren(parent)!!) {
            addChild(parent, child)
            if (source.hasChild(child)) {
                recursiveCopyFrom(source, child)
            }
        }
    }

    init {
        parentChildren = HashMap<T, MutableList<T>>()
        childParent = HashMap<T, T>()
    }
}