/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

/**
 * This class provides a skeletal implementation of the Selection interface to minimize the effort required to implement
 * this interface.
 */
abstract class AbstractSelection<E> : Selection<E> {
    override var isEnabled = true
        protected set(enabled) {
            if (isEnabled != enabled) {
                field = enabled
                val changes = emptySet<E>()
                notifySelectedChanged(SimpleSelectionEvent<E>(this, changes))
            }
        }
    private val listeners: MutableList<SelectionListener<E>>

    override fun addSelectionListener(listener: SelectionListener<E>) {
        listeners.add(listener)
    }

    override fun addWeakSelectionListener(listener: SelectionListener<E>) {
        val weakListener = WeakSelectionListener(listener)
        listeners.add(weakListener)
    }

    override fun removeSelectionListener(listener: SelectionListener<E>) {
        if (listener is WeakSelectionListener) {
            val removed = listeners.remove(listener)
//                assert(removed) { listener }
        } else {
            var toRemove: SelectionListener<E>? = null
            for (selectionListener in listeners) {
                var comparable: SelectionListener<E>?
                if (selectionListener is WeakSelectionListener) {
                    comparable = selectionListener.reference
                } else {
                    comparable = selectionListener
                }
                if (listener.equals(comparable)) {
                    toRemove = selectionListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
//                    assert(removed) { listener }
            }
        }
    }

    override fun removeSelectionListeners() {
        listeners.clear()
    }

    protected fun notifySelectedChanged(event: SelectionEvent<E>) {
        for (listener in listeners) {
            listener.selectionChanged(event)
        }
    }

    private inner class WeakSelectionListener(listener: SelectionListener<E>) : SelectionListener<E> {
        private val l_ref: WeakReference<SelectionListener<E>>
        override fun selectionChanged(event: SelectionEvent<E>) {
            val l: SelectionListener<E>? = reference
            if (l != null) {
                l.selectionChanged(event)
            } else {
                removeSelectionListener(this as SelectionListener<E>)
            }
        }

        val reference: SelectionListener<E>?
            get() = l_ref.get()

        override fun toString(): String {
            val l: SelectionListener<E>? = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                reference.toString()
            }
        }

        init {
            l_ref = WeakReference<SelectionListener<E>>(listener)
        }
    }

    init {
        listeners = CollectionFactory.copyOnWriteArrayList<SelectionListener<E>>()
    }
}