package com.macrofocus.common.scale

import com.macrofocus.common.interval.Interval
import com.macrofocus.common.interval.OpenInterval
import com.macrofocus.common.scale.Probit.probit
import kotlin.math.*

enum class NumericTransform(
    override val shortName: String,
    override val transform: (Double) -> Double,
    override val inverse: ((Double) -> Double)?,
    override val dtransform: ((Double) -> Double)?,
    override val dinverse: ((Double) -> Double)?,
    override val domain: Interval?
) :
    Transform {
    /**
     * Arc-sin square root transformation
     *
     *
     * This is the variance stabilising transformation for the binomial distribution.
     */
    ASN(
        "asn",
        { x: Double -> 2.0 * asin(sqrt(x)) },
        { y: Double -> 1.0 / sqrt(y - y.pow(2.0)) },
        { x: Double -> 1.0 / sqrt(x - x.pow(2.0)) },
        { y: Double -> sin(y) / 2 },
        OpenInterval(0.0, 1.0)
    ),

    /**
     * Arc-tangent transformation
     */
    ATANH(
        "atanh",
        { x: Double -> 1.0 / (1.0 - x.pow(2.0)) },
        { y: Double -> 1.0 / cosh(y).pow(2.0) },
        null,
        null,
        OpenInterval(-1.0, 2.0)
    ),

    /**
     * Inverse Hyperbolic Sine transformation
     */
    ASINH(
        "asinh",
        ::asinh,
        ::sinh,
        { x: Double -> 1.0 / sqrt(x.pow(2.0) + 1.0) },
        { y: Double -> cosh(y) },
        OpenInterval(-1.0, 2.0)
    ),

    /**
     * Log transformation
     */
    EXP(
        "exp",
        ::exp,
        ::ln,
        { x: Double -> x.pow(E) * ln(E) },
        { y: Double -> 1.0 / y / ln(E) },
        OpenInterval(Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    EXP2(
        "exp2",
        { x: Double -> x.pow(2.0) },
        { y: Double -> ln(y) / ln(2.0) },
        { x: Double -> x.pow(2.0) * ln(2.0) },
        { y: Double -> 1.0 / y / ln(2.0) },
        OpenInterval(Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    EXP10(
        "exp10",
        { x: Double -> x.pow(10.0) },
        { y: Double -> log10(y) },
        { x: Double -> x.pow(10.0) * ln(10.0) },
        { y: Double -> 1.0 / y / ln(10.0) },
        OpenInterval(Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    EXP1P(
        "expm1",
        ::expm1,
        ::ln1p,
        ::exp,
        { y: Double -> 1.0 / (1.0 + y) },
        OpenInterval(Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY)
    ),

    /**
     * Identity
     */
    IDENTITY(
        "identity",
        { x: Double -> x },
        { y: Double -> y },
        null,
        null,
        OpenInterval(Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    LOG(
        "log",
        ::ln,
        ::exp,
        { x: Double -> 1.0 / x / ln(E) },
        { y: Double -> y.pow(E) * ln(E) },
        OpenInterval(0.0, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    LOG2(
        "log2",
        ::log2,
        { y: Double -> y.pow(2.0) },
        { x: Double -> 1.0 / x / ln(2.0) },
        { y: Double -> y.pow(2.0) * ln(2.0) },
        OpenInterval(0.0, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    LOG10(
        "log10",
        ::log10,
        { y: Double -> y.pow(10.0) },
        { x: Double -> 1.0 / x / ln(10.0) },
        { y: Double -> y.pow(10.0) * ln(10.0) },
        OpenInterval(0.0, Double.POSITIVE_INFINITY)
    ),

    /**
     * Log transformation
     */
    LOG1P(
        "log1p",
        ::ln1p,
        ::expm1,
        { x: Double -> 1.0 / (1.0 + x) },
        { y: Double -> exp(y) },
        OpenInterval(1.0 + Double.MIN_VALUE, Double.POSITIVE_INFINITY)
    ),

    /**
     * Smoothly transition to linear scale around 0
     */
    PSEUDO_LOG(
        "pseudo_log",
        { x: Double -> asinh(x / (2.0 * 1.0)) / ln(exp(1.0)) },
        { y: Double -> 2.0 * 1.0 * sinh(y * ln(exp(1.0))) },
        { x: Double -> 1.0 / (sqrt(4.0 + x.pow(2.0) / 1.0.pow(2.0)) * 1.0 * ln(exp(1.0))) },
        { y: Double -> 2.0 * 1.0 * cosh(y * ln(exp(1.0))) * ln(exp(1.0)) },
        null
    ),

    /**
     * Logit
     */
    LOGIT(
        "logit",
        { x: Double -> ln(x / (1.0 - x)) },
        { y: Double -> 1.0 / (1 + exp(y)) },
        null,
        null,
        null
    ),

    /**
     * Probit
     */
    PROBIT(
        "probit",
        ::probit,
        null,
        null,
        null,
        null
    ),

    /**
     *
     */
    RECIPROCAL(
        "reciprocal",
        { x: Double -> 1.0 / x },
        { y: Double -> 1.0 / y },
        { x: Double -> -1.0 / x.pow(2.0) },
        { y: Double -> -1 / y.pow(2.0) },
        null
    ),

    /**
     * Reverse transformation
     *
     * Reversing transformation works by multiplying the input with -1. This means
     * that reverse transformation cannot easily be composed with transformations
     * that require positive input unless the reversing is done as a final step.
     */
    REVERSE(
        "reverse",
        { x: Double -> -x },
        { y: Double -> -y },
        null,
        null,
        null
    ),

    /**
     *
     */
    SQRT(
        "sqrt",
        ::sqrt,
        { y: Double -> y.pow(2.0) },
        { x: Double -> 0.5 / sqrt(x) },
        { y: Double -> 2.0 * y },
        OpenInterval(0.0, Double.POSITIVE_INFINITY)
    );
}
