/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.hierarchy

interface Hierarchy<T> {
    val root: T
    fun isRoot(node: T): Boolean
    fun getParent(child: T): T?
    fun hasChild(parent: T): Boolean
    fun getChildren(parent: T): Iterable<T>?
    fun getChildList(parent: T): List<T>?
    fun getChild(parent: T, index: Int): T
    fun getChildCount(parent: T): Int
    fun getIndexOfChild(parent: T, child: T): Int
    fun containsChild(parent: T, child: T): Boolean
    fun containsChild(child: T): Boolean
    fun preorderIterator(): Iterable<T>
    fun breadthFirstIterator(): Iterable<T>
    fun depthFirstIterator(): Iterable<T>
    fun leavesIterator(): Iterable<T>

    /**
     * Creates and returns an iterable that traverses the subhierarchy rooted at the give node in preorder.
     * The first node returned by the iterator's next() method is the given node.
     *
     * @param parent the root of the hierarchy to traverse
     * @return an iterable that traverses the subtree rooted at this node in preorder.
     */
    fun preorderIterator(parent: T): Iterable<T>?

    /**
     * Creates and returns an iterable that traverses the subhierarchy rooted at the give node in breadth-first order.
     * The first node returned by the iterator's next() method is the given node.
     *
     * @param parent the root of the hierarchy to traverse
     * @return an iterable that traverses the subtree rooted at this node in breadth-first order.
     */
    fun breadthFirstIterator(parent: T): Iterable<T>?

    /**
     * Creates and returns an iterable that traverses the subhierarchy rooted at the give node in depth-first order.
     * The first node returned by the iterator's next() method is the leftmost leaf.
     *
     * @param parent the root of the hierarchy to traverse
     * @return an iterable that traverses the subtree rooted at this node in depth-first order.
     */
    fun depthFirstIterator(parent: T): Iterable<T>?
    fun leavesIterator(parent: T): Iterable<T>?
    fun addHierarchyListener(listener: HierarchyListener<T>)
    fun addWeakHierarchyListener(listener: HierarchyListener<T>)
    fun removeHierarchyListener(listener: HierarchyListener<T>)
    fun removeHierarchyListeners()
    fun setNotifyListeners(enable: Boolean)
    val listeners: Iterable<Any?>?

    /**
     * Returns the path from the root, to get to this node.  The last
     * element in the path is this node.
     *
     * @return an array of objects giving the path, where the
     * first element in the path is the root and the last
     * element is this node.
     */
    fun getPath(node: T): List<T>?
    fun getPathToRoot(child: T): Array<Any?>?
    fun notifyHierarchyNodeChanged(child: T, parent: T, index: Int, isAdjusting: Boolean)
    val depth: Int
    fun getLevel(node: T): Int
    fun isLeaf(node: T): Boolean
    fun getLeafCount(node: T): Int
    fun getFirstLeaf(node: T): T
    fun getLastLeaf(node: T): T
    fun getNextLeaf(node: T): T?
    fun getPreviousLeaf(node: T): T?
}