/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.filter

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

/**
 * This class provides a skeletal implementation of the Filter interface to minimize the effort required to implement
 * this interface.
 */
abstract class AbstractFilter<E> : MutableFilter<E> {
    override var isEnabled = true
    private val listeners: MutableList<FilterListener<E>>
//    fun isEnabled(): Boolean {
//        return enabled
//    }
//
//    override fun setEnabled(enabled: Boolean) {
//        if (this.enabled != enabled) {
//            this.enabled = enabled
//        }
//    }

    override fun addFilterListener(listener: FilterListener<E>) {
        listeners.add(listener)
    }

    override fun addWeakFilterListener(listener: FilterListener<E>) {
        listeners.add(WeakFilterListener(listener))
    }

    override fun removeFilterListener(listener: FilterListener<E>) {
        if (listener is WeakFilterListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: FilterListener<E>? = null
            for (filterListener in listeners) {
                var comparable: FilterListener<E>
                comparable = if (filterListener is WeakFilterListener) {
                    filterListener.reference
                } else {
                    filterListener
                }
                if (listener.equals(comparable)) {
                    toRemove = filterListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    override fun removeFilterListeners() {
        listeners.clear()
    }

    protected fun notifyFilteredChanged(event: FilterEvent<E>) {
        for (listener in listeners) {
            listener.filterChanged(event)
        }
    }

    private inner class WeakFilterListener(listener: FilterListener<E>) : FilterListener<E> {
        private val l_ref: WeakReference<FilterListener<E>>
        override fun filterChanged(event: FilterEvent<E>) {
            val l: FilterListener<E> = reference
            if (l != null) {
                l.filterChanged(event)
            } else {
                removeFilterListener(this as FilterListener<E>)
            }
        }

        val reference: FilterListener<E>
            get() = l_ref.get()

        init {
            l_ref = WeakReference<FilterListener<E>>(listener)
        }
    }

    companion object {
        private const val serialVersionUID = 6388032080849307737L
    }

    init {
        listeners = CollectionFactory.copyOnWriteArrayList<FilterListener<E>>()
    }
}