/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.properties

/**
 * Default data model for a collection of properties.
 */
open class OverrideProperties(
    private val defaults: MutableProperties<String?>,
    private val overrideDefaults: MutableProperty<Boolean>
) : AbstractProperties<String?>(), MutableProperties<String?> {
    private val propertyMap: MutableMap<String?, OverrideProperty<Any?>?> = HashMap<String?, OverrideProperty<Any?>?>()
    private val listenerMap: MutableMap<String?, PropertyListener<Any?>> = HashMap<String?, PropertyListener<Any?>>()

    override operator fun iterator(): Iterator<String?> {
        return defaults.iterator()
    }

    /**
     * Gets whether the default settings are overridden by these settings
     *
     * @return true if this override the default settings, false otherwise
     */
    fun isOverrideDefaults(): Boolean {
        return overrideDefaults.value
    }

    /**
     * Sets whether the default settings should be overridden by these settings
     *
     * @param overrideDefaults true to override the default settings, false otherwise
     */
    fun setOverrideDefaults(overrideDefaults: Boolean) {
        this.overrideDefaults.value = overrideDefaults
    }

    override fun <T> createProperty(name: String?, value: T): MutableProperty<T> {
        throw UnsupportedOperationException()
        //        MutableProperty<T> property = SimpleProperty.newInstance(value);
//        return addProperty(name, property);
    }

    override fun <T> addProperty(name: String?, property: MutableProperty<T>): MutableProperty<T> {
        throw UnsupportedOperationException()
        //        assert !propertyMap.containsKey(name): "Property " + name + " already exists";
//        propertyMap.put(name, property);
//        registerPropertyListener(name, property);
//        final NamedPropertyListener listener = new NamedPropertyListener(name);
//        property.addPropertyListener(listener);
//        listenerMap.put(name, listener);
//        return property;
    }

    override fun <T> replaceProperty(name: String?, property: MutableProperty<T>): MutableProperty<T> {
        throw UnsupportedOperationException()
        //        assert propertyMap.containsKey(name);
//
//        MutableProperty<T> oldProperty = propertyMap.get(name);
//        Object oldValue = oldProperty.getValue();
//        oldProperty.removePropertyListener(listenerMap.get(name));
//        propertyMap.put(name, property);
//        final NamedPropertyListener listener = new NamedPropertyListener(name);
//        property.addPropertyListener(listener);
//        listenerMap.put(name, listener);
//
//        notifyPropertyChanged(name, new PropertyEvent(oldValue, property.getValue()));
//
//        return property;
    }

    override fun getValue(name: String?): Any? {
        return if (overrideDefaults.value && propertyMap!!.containsKey(name)) propertyMap[name]!!.value else defaults!!.getValue(name)
    }

    override fun setValue(name: String?, value: Any?) {
        propertyMap!![name]!!.value = value
        //        if(overrideDefaults.getValue()) {
//            if (propertyMap.containsKey(name)) {
//                final Object old = propertyMap.get(name).getValue();
//                if (value != old) {
//                    if(value != null) {
//                        propertyMap.get(name).setValue(value);
//                        notifyPropertyChanged(name, new PropertyEvent(old, value));
//                    } else {
//                        propertyMap.remove(name);
//                        notifyPropertyChanged(name, new PropertyEvent(old, defaults.getValue(name)));
//                    }
//                }
//            } else {
//                if(value != null) {
//                    final Object old = null;
//                    propertyMap.get(name).setValue(value);
//                    notifyPropertyChanged(name, new PropertyEvent(old, value));
//                }
//            }
//        } else {
//            defaults.setValue(name, value);
//        }
    }

    override fun getProperty(name: String?): OverrideProperty<Any?> {
        return if (propertyMap!!.containsKey(name)) {
            propertyMap[name]!!
        } else {
            val property: OverrideProperty<Any?> = createProperty(name, overrideDefaults, defaults!!.getProperty(name))
            registerPropertyListener(name, property)
            propertyMap[name] = property
            property
        }
    }

    protected fun registerPropertyListener(name: String?, property: MutableProperty<Any?>) {
        property.addPropertyListener(object : PropertyListener<Any?> {
            override fun propertyChanged(event: PropertyEvent<Any?>) {
                notifyPropertyChanged(name, PropertyEvent(event.oldValue, event.newValue))
            }
        })
    }

    protected open fun createProperty(
        name: String?,
        overrideDefaults: MutableProperty<Boolean>,
        defaultProperty: MutableProperty<Any?>
    ): OverrideProperty<Any?> {
        return OverrideProperty(overrideDefaults, defaultProperty)
    }

    fun reset() {
        for (property in propertyMap!!.values) {
            if (property is OverrideProperty) {
                (property as OverrideProperty?)!!.reset()
            }
        }
    }

    private inner class NamedPropertyListener private constructor(private val name: String?) : PropertyListener<Any?> {
        override fun propertyChanged(event: PropertyEvent<Any?>) {
            notifyPropertyChanged(name, event)
        }

    }

    companion object {
        var PROPERTY_OVERRIDE_DEFAULTS: String? = "overrideDefaults"
    }

    init {
//        defaults.addPropertiesListener(new PropertiesListener<String>() {
//            @Override
//            public void propertyChanged(String name, PropertyEvent event) {
//                if(!overrideDefaults.getValue() || !propertyMap.containsKey(name)) {
//                    notifyPropertyChanged(name, new PropertyEvent(event.getOldValue(), event.getNewValue()));
//                }
//            }
//        });

//        overrideDefaults.addPropertyListener(new PropertyListener<Boolean>() {
//            @Override
//            public void propertyChanged(PropertyEvent<Boolean> event) {
//                notifyPropertyChanged(PROPERTY_OVERRIDE_DEFAULTS, new PropertyEvent(event.getOldValue(), event.getNewValue()));
//
//                if(overrideDefaults.getValue()) {
//                    for(Map.Entry<String, MutableProperty> entry: propertyMap.entrySet()) {
//                        final String key = entry.getKey();
//                        final Object oldValue = defaults.getValue(key);
//                        final Object newValue = entry.getValue().getValue();
//                        if(oldValue != newValue) {
//                            notifyPropertyChanged(key, new PropertyEvent(oldValue, newValue));
//                        }
//                    }
//                } else {
//                    for(Map.Entry<String, MutableProperty> entry: propertyMap.entrySet()) {
//                        final String key = entry.getKey();
//                        final Object newValue = defaults.getValue(key);
//                        final Object oldValue = entry.getValue().getValue();
//                        if(oldValue != newValue) {
//                            notifyPropertyChanged(key, new PropertyEvent(oldValue, newValue));
//                        }
//                    }
//                }
//
//            }
//        });
    }
}