package com.macrofocus.common.units

import kotlin.math.absoluteValue
import kotlin.math.roundToLong

/**
 * A quantity with a unit type
 */
class Quantity<T: Unit>(val amount: Double, val unit: T): Comparable<Quantity<T>> {
    /**
     * Convert this type into another compatible type.
     * Type must share parent
     * (eg Mile into Kilometer, because they both are made from Distance)
     */
    infix fun <A: T> into(unit: A): Quantity<A> {
        val baseUnit = this.unit.convertToBaseUnit(amount)
        return Quantity(unit.convertFromBaseUnit(baseUnit), unit)
    }

    /**
     * Add another compatible quantity to this one
     */
    operator fun plus(quantity: Quantity<T>): Quantity<T> {
        val converted = quantity.into(unit).amount
        return Quantity(this.amount + converted, this.unit)
    }

    /**
     * Subtract a compatible quantity from this one
     */
    operator fun minus(quantity: Quantity<T>): Quantity<T> {
        val converted = quantity.into(unit).amount
        return Quantity(this.amount - converted, this.unit)
    }

    /**
     * Divide this by another compatible quantity, gives back a ratio of the sizes
     */
    operator fun div(other: Quantity<T>) = unit.convertToBaseUnit(amount) / other.unit.convertToBaseUnit(other.amount)

    /**
     * Divide this by an incompatible quantity, creating a Quantity<QuotientUnit<T, R>>
     */
    operator fun <R: Unit> div(quantity: Quantity<R>) = Quantity(amount / quantity.amount, unit / quantity.unit)

    /**
     * Multiply this by a quantity to get a product quantity, Quantity<ProductQuantity<T, R>>
     */
    operator fun <R: Unit> times(quantity: Quantity<R>) = Quantity(amount * quantity.amount, unit * quantity.unit)

    /**
     * Multiply this by a scalar value, used for things like "double this distance",
     * "1.5 times the speed", etc
     */
    operator fun times(other: Number) = Quantity(amount * other.toDouble(), unit)

    /**
     * Divide this by a scalar, used for things like "halve the speed"
     */
    operator fun div(other: Number) = Quantity(amount / other.toDouble(), unit)

    /**
     * Compare this value with another quantity - which must have the same type
     * Units are converted before comparison
     */
    override fun compareTo(other: Quantity<T>): Int {
        return this.into(other.unit).amount.compareTo(other.amount)
    }

    val EPSILON = 1e-10

    /**
     * Check if quantities are equal, not sure if there's a nicer way of checking this
     * partly due to type erasure
     */
    override fun equals(other: Any?): Boolean {
        return this === other || (other is Quantity<*> &&
                other.unit::class == unit::class &&
                (unit.convertToBaseUnit(amount) - other.unit.convertToBaseUnit(other.amount)).absoluteValue < EPSILON)
    }

    override fun hashCode(): Int {
        return (unit.convertFromBaseUnit(amount) / EPSILON.roundToLong()).hashCode()
    }

    override fun toString(): String {
        return "$amount ${unit.symbol}"
    }
}

/**
 * Helpers for converting numbers into quantities
 */
infix fun <T: Unit> Number.into(unit: T) = Quantity(this.toDouble(), unit)
operator fun <T: Unit> Number.times(unit: T) = this into unit
operator fun <T: Unit> T.times(value: Number) = value into this
operator fun <T: Unit> T.invoke(value: Number) = value into this
/**
 * Inverse of Quantity.times(value: Number)
 * Haven't implemented division due to no reciprocal unit
 */
operator fun <T: Unit> Number.times(quantity: Quantity<T>) = quantity * this

