/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream

/**
 * Server-side implementation of JsonBoolean.
 */
class JavaJsonBoolean(override var boolean: Boolean) : JavaJsonValue(), JsonBoolean {
    override fun asBoolean(): Boolean {
        return boolean
    }

    override fun asNumber(): Double {
        return if (boolean) 1.0 else 0.0
    }

    override fun asString(): String {
        return java.lang.Boolean.toString(boolean)
    }

    override val `object`: Any
        get() = boolean
    override val type: com.macrofocus.common.json.JsonType
        get() = JsonType.BOOLEAN

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JavaJsonValue).`object`
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        visitor.visit(boolean, ctx)
    }

    @Throws(IllegalStateException::class)
    override fun toJson(): String {
        return boolean.toString()
    }

    @Throws(IOException::class, ClassNotFoundException::class)
    private fun readObject(stream: ObjectInputStream) {
        val instance: JavaJsonBoolean = parseJson(stream)
        boolean = instance.boolean
    }

    @Throws(IOException::class)
    private fun writeObject(stream: ObjectOutputStream) {
        stream.writeObject(toJson())
    }

    companion object {
        private const val serialVersionUID = 1L
    }
}