/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream
import java.util.*

/**
 * Server-side implementation of JsonObject.
 */
class JavaJsonObject(factory: JsonFactory) : JavaJsonValue(), JsonObject {
    private var factory: JsonFactory

    private var map: MutableMap<String, JsonValue?> = LinkedHashMap<String, JsonValue?>()
    override fun asBoolean(): Boolean {
        return true
    }

    override fun asNumber(): Double {
        return Double.NaN
    }

    override fun asString(): String {
        return "[object Object]"
    }

    override operator fun <T : JsonValue?> get(key: String): T {
        return map[key] as T
    }

    override fun getArray(key: String): JsonArray? {
        return get(key) as JsonArray?
    }

    override fun getBoolean(key: String): Boolean {
        return (get(key) as JsonBoolean).boolean
    }

    override fun getNumber(key: String): Double {
        return (get(key) as JsonNumber).number
    }

    override fun getObject(key: String): JsonObject? {
        return get(key) as JsonObject?
    }

    override val `object`: Any
        get() {
            val obj: MutableMap<String, Any> = HashMap()
            for ((key, value) in map) {
                obj[key] = (value as JavaJsonValue).`object`
            }
            return obj
        }

    override fun getString(key: String): String {
        return (get(key) as JsonString).string
    }

    override val type: com.macrofocus.common.json.JsonType
        get() = JsonType.OBJECT

    override fun hasKey(key: String): Boolean {
        return map.containsKey(key)
    }

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JavaJsonValue).`object`
    }

    override fun keys(): Array<String> {
        return map.keys.toTypedArray()
    }

    override fun put(key: String, value: JsonValue?) {
        var value: JsonValue? = value
        if (value == null) {
            value = factory.createNull()
        }
        map[key] = value
    }

    override fun put(key: String, value: String) {
        put(key, factory.create(value))
    }

    override fun put(key: String, value: Double) {
        put(key, factory.create(value))
    }

    override fun put(key: String, bool: Boolean) {
        put(key, factory.create(bool))
    }

    override fun remove(key: String?) {
        map.remove(key)
    }

    operator fun set(key: String, value: JsonValue?) {
        put(key, value)
    }

    override fun toJson(): String {
        return JsonUtil.stringify(this)
    }

    override fun toString(): String {
        return toJson()
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        if (visitor.visit(this, ctx)) {
            val objCtx = JsonObjectContext(this)
            for (key in stringifyOrder(keys())) {
                objCtx.currentKey = key
                if (visitor.visitKey(objCtx.currentKey, objCtx)) {
                    visitor.accept(get(key), objCtx)
                    objCtx.isFirst = false
                }
            }
        }
        visitor.endVisit(this, ctx)
    }

    @Throws(IOException::class, ClassNotFoundException::class)
    private fun readObject(stream: ObjectInputStream) {
        val instance: JavaJsonObject = parseJson(stream)
        factory = Json.instance()
        map = instance.map
    }

    @Throws(IOException::class)
    private fun writeObject(stream: ObjectOutputStream) {
        stream.writeObject(toJson())
    }

    companion object {
        private const val serialVersionUID = 1L
        private fun stringifyOrder(keys: Array<String>): List<String> {
            val toReturn: MutableList<String> = java.util.ArrayList<String>()
            val nonNumeric: MutableList<String> = java.util.ArrayList<String>()
            for (key in keys) {
                if (key.matches("\\d+".toRegex())) {
                    toReturn.add(key)
                } else {
                    nonNumeric.add(key)
                }
            }
            java.util.Collections.sort<String>(toReturn)
            toReturn.addAll(nonNumeric)
            return toReturn
        }
    }

    init {
        this.factory = factory
    }
}