/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

/**
 * Implementation of parsing a JSON string into instances of [ ].
 */
internal class JsonTokenizer(serverJsonFactory: JsonFactory, json: String) {
    private val jsonFactory: JsonFactory
    private val lenient = true
    private var pushBackBuffer = INVALID_CHAR
    private val json: String
    private var position = 0
    fun back(c: Char) {
        pushBackBuffer = c.toInt()
    }

    fun back(c: Int) {
        back(c.toChar())
    }

    operator fun next(): Int {
        if (pushBackBuffer != INVALID_CHAR) {
            val c = pushBackBuffer
            pushBackBuffer = INVALID_CHAR
            return c
        }
        return if (position < json.length) json[position++].toInt() else INVALID_CHAR
    }

    @Throws(JsonException::class)
    fun next(n: Int): String {
        if (n == 0) {
            return ""
        }
        val buffer = CharArray(n)
        var pos = 0
        if (pushBackBuffer != INVALID_CHAR) {
            buffer[0] = pushBackBuffer.toChar()
            pos = 1
            pushBackBuffer = INVALID_CHAR
        }
        var len: Int = 0
        while (pos < n && read(buffer, pos, n - pos).also { len = it } != -1) {
            pos += len
        }
        if (pos < n) {
            throw JsonException("TODO" /* TODO(knorton): Add message. */)
        }
        return String(buffer)
    }

    fun nextNonWhitespace(): Int {
        while (true) {
            val c = next()
            if (!java.lang.Character.isSpace(c.toChar())) {
                return c
            }
        }
    }

    @Throws(JsonException::class)
    fun nextString(startChar: Int): String {
        val buffer = StringBuilder()
        var c = next()
        assert(c == '"'.toInt() || lenient && c == '\''.toInt())
        while (true) {
            c = next()
            when (c) {
                '\r'.toInt(), '\n'.toInt() -> throw JsonException("")
                INVALID_CHAR -> throw JsonException("Invalid string: closing $startChar is not found")
                '\\'.toInt() -> {
                    c = next()
                    when (c) {
                        'b'.toInt() -> buffer.append('\b')
                        't'.toInt() -> buffer.append('\t')
                        'n'.toInt() -> buffer.append('\n')
//                        'f'.toInt() -> buffer.append('\f') // ToDo: invalid!
                        'r'.toInt() -> buffer.append('\r')
                        'u'.toInt() -> buffer.append(next(4).toInt(16).toChar())
                        else -> buffer.append(c.toChar())
                    }
                }
                else -> {
                    if (c == startChar) {
                        return buffer.toString()
                    }
                    buffer.append(c.toChar())
                }
            }
        }
    }

    fun nextUntilOneOf(chars: String): String {
        val buffer = StringBuilder()
        var c = next()
        while (c != INVALID_CHAR) {
            if (java.lang.Character.isSpace(c.toChar()) || chars.indexOf(c.toChar()) >= 0) {
                back(c)
                break
            }
            buffer.append(c.toChar())
            c = next()
        }
        return buffer.toString()
    }

    @Throws(JsonException::class)
    fun <T : JsonValue?> nextValue(): T {
        val c = nextNonWhitespace()
        back(c)
        return when (c) {
            '"'.toInt(), '\''.toInt() -> jsonFactory.create(nextString(c)) as T
            '{'.toInt() -> parseObject() as T
            '['.toInt() -> parseArray() as T
            else -> getValueForLiteral(nextUntilOneOf(STOPCHARS)) as T
        }
    }

    @Throws(JsonException::class)
    fun parseArray(): JsonArray {
        val array: JsonArray = jsonFactory.createArray()
        var c = nextNonWhitespace()
        assert(c == '['.toInt())
        while (true) {
            c = nextNonWhitespace()
            when (c) {
                ']'.toInt() -> return array
                else -> {
                    back(c)
                    array.set(array.length(), nextValue<JsonValue>())
                    val d = nextNonWhitespace()
                    when (d) {
                        ']'.toInt() -> return array
                        ','.toInt() -> {
                        }
                        else -> throw JsonException("Invalid array: expected , or ]")
                    }
                }
            }
        }
    }

    @Throws(JsonException::class)
    fun parseObject(): JsonObject {
        val `object`: JsonObject = jsonFactory.createObject()
        var c = nextNonWhitespace()
        if (c != '{'.toInt()) {
            throw JsonException(
                "Payload does not begin with '{'.  Got " + c + "("
                        + java.lang.Character.valueOf(c.toChar()) + ")"
            )
        }
        while (true) {
            c = nextNonWhitespace()
            when (c) {
                '}'.toInt() ->           // We're done.
                    return `object`
                '"'.toInt(), '\''.toInt() -> {
                    back(c)
                    // Ready to start a key.
                    val key = nextString(c)
                    if (nextNonWhitespace() != ':'.toInt()) {
                        throw JsonException(
                            "Invalid object: expecting \":\""
                        )
                    }
                    // TODO(knorton): Make sure this key is not already set.
                    `object`.put(key, nextValue<JsonValue>())
                    when (nextNonWhitespace()) {
                        ','.toInt() -> {
                        }
                        '}'.toInt() -> return `object`
                        else -> throw JsonException(
                            "Invalid object: expecting } or ,"
                        )
                    }
                }
                ','.toInt() -> {
                }
                else -> if (lenient && (java.lang.Character.isDigit(c.toChar()) || java.lang.Character.isLetterOrDigit(c.toChar()))) {
                    val keyBuffer = StringBuilder()
                    keyBuffer.append(c)
                    while (true) {
                        c = next()
                        if (java.lang.Character.isDigit(c.toChar()) || java.lang.Character.isLetterOrDigit(c.toChar())) {
                            keyBuffer.append(c)
                        } else {
                            back(c)
                            break
                        }
                    }
                    if (nextNonWhitespace() != ':'.toInt()) {
                        throw JsonException(
                            "Invalid object: expecting \":\""
                        )
                    }
                    // TODO(knorton): Make sure this key is not already set.
                    `object`.put(keyBuffer.toString(), nextValue<JsonValue>())
                    when (nextNonWhitespace()) {
                        ','.toInt() -> {
                        }
                        '}'.toInt() -> return `object`
                        else -> throw JsonException(
                            "Invalid object: expecting } or ,"
                        )
                    }
                } else {
                    throw JsonException("Invalid object: ")
                }
            }
        }
    }

    @Throws(JsonException::class)
    private fun getNumberForLiteral(literal: String): JsonNumber {
        return try {
            jsonFactory.create(literal.toDouble())
        } catch (e: NumberFormatException) {
            throw JsonException("Invalid number literal: $literal")
        }
    }

    @Throws(JsonException::class)
    private fun getValueForLiteral(literal: String): JsonValue {
        if ("" == literal) {
            throw JsonException("Missing value")
        }
        if ("null" == literal || "undefined" == literal) {
            return jsonFactory.createNull()
        }
        if ("true" == literal) {
            return jsonFactory.create(true)
        }
        if ("false" == literal) {
            return jsonFactory.create(false)
        }
        val c = literal[0]
        if (c == '-' || java.lang.Character.isDigit(c)) {
            return getNumberForLiteral(literal)
        }
        throw JsonException("Invalid literal: \"$literal\"")
    }

    private fun read(buffer: CharArray, pos: Int, len: Int): Int {
        val maxLen: Int = java.lang.Math.min(json.length - position, len)
        val src = json.substring(position, position + maxLen)
        val result = src.toCharArray()
        java.lang.System.arraycopy(result, 0, buffer, pos, maxLen)
        position += maxLen
        return maxLen
    }

    companion object {
        private const val INVALID_CHAR = -1
        private const val STOPCHARS = ",:]}/\\\"[{;=#"
    }

    init {
        jsonFactory = serverJsonFactory
        this.json = json
    }
}