/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

/**
 * A visitor for JSON objects. For each unique JSON datatype, a callback is
 * invoked with a [com.macrofocus.common.json.impl.JsonContext]
 * that can be used to replace a value or remove it. For Object and Array
 * types, the [.visitKey] and [.visitIndex] methods are invoked
 * respectively for each contained value to determine if they should be
 * processed or not. Finally, the visit methods for Object and Array types
 * returns a boolean that determines whether or not to process its contained
 * values.
 */
open class JsonVisitor {
    private inner class ImmutableJsonContext(node: JsonValue) : JsonContext(node) {
        override fun removeMe() {
            immutableError()
        }

        override fun replaceMe(d: Double) {
            immutableError()
        }

        override fun replaceMe(d: String) {
            immutableError()
        }

        override fun replaceMe(d: Boolean) {
            immutableError()
        }

        override fun replaceMe(value: JsonValue?) {
            immutableError()
        }

        private fun immutableError() {
            throw UnsupportedOperationException("Immutable context")
        }
    }

    fun accept(node: JsonValue) {
        accept(node, ImmutableJsonContext(node))
    }

    /**
     * Accept array or object type and visit its members.
     */
    fun accept(node: JsonValue?, ctx: JsonContext) {
        if (node == null) {
            return
        }
        (node as JavaJsonValue).traverse(this, ctx)
    }

    /**
     * Called after every element of array has been visited.
     */
    open fun endVisit(array: JsonArray, ctx: JsonContext) {}

    /**
     * Called after every field of an object has been visited.
     * @param object
     * @param ctx
     */
    open fun endVisit(`object`: JsonObject, ctx: JsonContext) {}

    /**
     * Called for JS numbers present in a JSON object.
     */
    open fun visit(number: Double, ctx: JsonContext) {}

    /**
     * Called for JS strings present in a JSON object.
     */
    open fun visit(string: String?, ctx: JsonContext) {}

    /**
     * Called for JS boolean present in a JSON object.
     */
    open fun visit(bool: Boolean, ctx: JsonContext) {}

    /**
     * Called for JS arrays present in a JSON object. Return true if array
     * elements should be visited.
     * @param array a JS array
     * @param ctx a context to replace or delete the array
     * @return true if the array elements should be visited
     */
    open fun visit(array: JsonArray, ctx: JsonContext): Boolean {
        return true
    }

    /**
     * Called for JS objects present in a JSON object. Return true if object
     * fields should be visited.
     * @param object a Json object
     * @param ctx a context to replace or delete the object
     * @return true if object fields should be visited
     */
    open fun visit(`object`: JsonObject, ctx: JsonContext): Boolean {
        return true
    }

    /**
     * Return true if the value for a given array index should be visited.
     * @param index an index in a JSON array
     * @param ctx a context object used to delete or replace values
     * @return true if the value associated with the index should be visited
     */
    open fun visitIndex(index: Int, ctx: JsonContext): Boolean {
        return true
    }

    /**
     * Return true if the value for a given object key should be visited.
     * @param key a key in a JSON object
     * @param ctx a context object used to delete or replace values
     * @return true if the value associated with the key should be visited
     */
    open fun visitKey(key: String?, ctx: JsonContext): Boolean {
        return true
    }

    /**
     * Called for nulls present in a JSON object.
     */
    open fun visitNull(ctx: JsonContext) {}
}