/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.collection

fun toString(a: Array<Any?>?): String? {
    if (a == null) return "null"
    val iMax = a.size - 1
    if (iMax == -1) return "[]"
    val b: StringBuilder = StringBuilder()
    b.append('[')
    var i = 0
    while (true) {
        b.append(a[i].toString())
        if (i == iMax) return b.append(']').toString()
        b.append(", ")
        i++
    }
}

fun <T> sort(a: Array<T>, c: Comparator<in T>) {
    a.sortWith(c)
}

/**
 * Searches the specified array of ints for the specified value using the
 * binary search algorithm.  The array must be sorted (as
 * by the [.sort] method) prior to making this call.  If it
 * is not sorted, the results are undefined.  If the array contains
 * multiple elements with the specified value, there is no guarantee which
 * one will be found.
 *
 * @param a the array to be searched
 * @param key the value to be searched for
 * @return index of the search key, if it is contained in the array;
 * otherwise, `(-(*insertion point*) - 1)`.  The
 * *insertion point* is defined as the point at which the
 * key would be inserted into the array: the index of the first
 * element greater than the key, or `a.length` if all
 * elements in the array are less than the specified key.  Note
 * that this guarantees that the return value will be &gt;= 0 if
 * and only if the key is found.
 */
fun binarySearch(a: IntArray, key: Int): Int {
    return binarySearch0(a, 0, a.size, key)
}

private fun binarySearch0(
    a: IntArray, fromIndex: Int, toIndex: Int,
    key: Int
): Int {
    var low = fromIndex
    var high = toIndex - 1
    while (low <= high) {
        val mid = low + high ushr 1
        val midVal = a[mid]
        if (midVal < key) low = mid + 1 else if (midVal > key) high = mid - 1 else return mid // key found
    }
    return -(low + 1) // key not found.
}

fun copyOf(original: IntArray, newLength: Int): IntArray {
    val copy = IntArray(newLength)
    arraycopy(
        original, 0, copy, 0,
        kotlin.math.min(original.size, newLength)
    )
    return copy
}

fun copyOfRange(original: IntArray, from: Int, to: Int): IntArray {
    val newLength = to - from
    require(newLength >= 0) { "$from > $to" }
    val copy = IntArray(newLength)
    arraycopy(
        original, from, copy, 0,
        kotlin.math.min(original.size - from, newLength)
    )
    return copy
}

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun <T> arraycopy(src: Array<out T>, srcPos: Int, dst: Array<T>, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: BooleanArray, srcPos: Int, dst: BooleanArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: LongArray, srcPos: Int, dst: LongArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: ByteArray, srcPos: Int, dst: ByteArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: ShortArray, srcPos: Int, dst: ShortArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: CharArray, srcPos: Int, dst: CharArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: IntArray, srcPos: Int, dst: IntArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: FloatArray, srcPos: Int, dst: FloatArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }

/** Copies [size] elements of [src] starting at [srcPos] into [dst] at [dstPos]  */
fun arraycopy(src: DoubleArray, srcPos: Int, dst: DoubleArray, dstPos: Int, size: Int): Unit =
    run { src.copyInto(dst, dstPos, srcPos, srcPos + size) }
