package com.macrofocus.common.scale

import kotlin.math.ln
import kotlin.math.sqrt

/**
 * Provides a [.probit] method to convert from the (0.0,1.0) domain to a Gaussian range.
 * Uses an algorithm by Peter John Acklam, as implemented by Sherali Karimov.
 * Free to use, but you should credit Acklam and Karimov appropriately.
 */
object Probit {
    private const val LOW = 0.02425
    private const val HIGH = 1.0 - LOW

    // Coefficients in rational approximations.
    private const val A0 = -3.969683028665376e+01
    private const val A1 = 2.209460984245205e+02
    private const val A2 = -2.759285104469687e+02
    private const val A3 = 1.383577518672690e+02
    private const val A4 = -3.066479806614716e+01
    private const val A5 = 2.506628277459239e+00

    private const val B0 = -5.447609879822406e+01
    private const val B1 = 1.615858368580409e+02
    private const val B2 = -1.556989798598866e+02
    private const val B3 = 6.680131188771972e+01
    private const val B4 = -1.328068155288572e+01

    private const val C0 = -7.784894002430293e-03
    private const val C1 = -3.223964580411365e-01
    private const val C2 = -2.400758277161838e+00
    private const val C3 = -2.549732539343734e+00
    private const val C4 = 4.374664141464968e+00
    private const val C5 = 2.938163982698783e+00

    private const val D0 = 7.784695709041462e-03
    private const val D1 = 3.224671290700398e-01
    private const val D2 = 2.445134137142996e+00
    private const val D3 = 3.754408661907416e+00

    /**
     * A way of taking a double in the (0.0, 1.0) range and mapping it to a Gaussian or normal distribution, so high
     * inputs correspond to high outputs, and similarly for the low range. This is centered on 0.0 and its standard
     * deviation seems to be 1.0 (the same as [Random.nextGaussian]). It is slightly faster (about a 10%
     * increase in throughput, with comparable overhead for both methods) than the approach used by
     * [Random.nextGaussian], even when both use a faster RNG algorithm and when both values produced by
     * nextGaussian() are used.
     * <br></br>
     * This uses an algorithm by Peter John Acklam, as implemented by Sherali Karimov.
     * [Source](https://web.archive.org/web/20150910002142/http://home.online.no/~pjacklam/notes/invnorm/impl/karimov/StatUtil.java).
     * [Information on the algorithm](https://web.archive.org/web/20151030215612/http://home.online.no/~pjacklam/notes/invnorm/).
     * @param d should be between 0 and 1, exclusive, but other values are tolerated (they return infinite results)
     * @return a normal-distributed double centered on 0.0
     */
    // This can legitimately return infinite doubles, which it produces with zero division.
    fun probit(d: Double): Double {
        if (d <= 0 || d >= 1) {
            return (d - 0.5) / 0.0
        } else if (d < LOW) {
            val q = sqrt(-2 * ln(d))
            return (((((C0 * q + C1) * q + C2) * q + C3) * q + C4) * q + C5) / ((((D0 * q + D1) * q + D2) * q + D3) * q + 1)
        } else if (HIGH < d) {
            val q = sqrt(-2 * ln(1 - d))
            return -(((((C0 * q + C1) * q + C2) * q + C3) * q + C4) * q + C5) / ((((D0 * q + D1) * q + D2) * q + D3) * q + 1)
        } else {
            val q = d - 0.5
            val r = q * q
            return (((((A0 * r + A1) * r + A2) * r + A3) * r + A4) * r + A5) * q / (((((B0 * r + B1) * r + B2) * r + B3) * r + B4) * r + 1)
        }
    }
}
