package com.macrofocus.common.crossplatform

import com.macrofocus.common.concurrent.ExecutorService
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.common.timer.SwingTimer
import java.util.*
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.ThreadFactory
import java.util.concurrent.ThreadPoolExecutor
import java.util.concurrent.TimeUnit
import kotlin.reflect.KClass
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

actual class CPHelper {
    val nAvailableProcessors : Int by lazy { Math.max(Runtime.getRuntime().availableProcessors(), 1) }
    val visualizationExecutorService: ExecutorService? by lazy { newFixedThreadPool("VisualizationExecutorService", 0, nAvailableProcessors) }

    actual companion object {
        actual val instance = CPHelper()
    }

    actual fun newFixedThreadPool(
        name: String,
        corePoolSize: Int,
        maximumPoolSize: Int
    ): ExecutorService? {
//        return Executors.newFixedThreadPool(corePoolSize);
        val threadPoolExecutor = ThreadPoolExecutor(corePoolSize, maximumPoolSize,
            20L, TimeUnit.SECONDS,
            LinkedBlockingQueue(),
            object : ThreadFactory {
                val group = if (System.getSecurityManager() != null) System.getSecurityManager().threadGroup else Thread.currentThread().threadGroup
                override fun newThread(r: Runnable): Thread {
                    val t = Thread(
                        group, r,
                        name,
                        0
                    )
                    if (t.isDaemon) t.isDaemon = false
                    if (t.priority != Thread.NORM_PRIORITY) t.priority = Thread.NORM_PRIORITY
                    return t
                }
            })
        return ExecutorService(threadPoolExecutor)
    }

    actual fun visualizationExecutorService(): ExecutorService? {
        return visualizationExecutorService
    }

    actual fun createTimer(
        name: String,
        delay: Int,
        postponing: Boolean
    ): CPTimer {
        return SwingTimer(name, delay, 10000, postponing)
    }

    actual fun createTimer(
        name: String,
        delay: Int,
        postponing: Boolean,
        timerListener: CPTimerListener
    ): CPTimer {
        val timer = createTimer(name, delay, postponing)
        timer.addTimerListener(timerListener)
        return timer
    }

    /**
     * Causes the currently executing thread to sleep (temporarily cease
     * execution) for the specified number of milliseconds, subject to
     * the precision and accuracy of system timers and schedulers. The thread
     * does not lose ownership of any monitors.
     *
     * @param  millis
     * the length of time to sleep in milliseconds
     *
     * @throws  IllegalArgumentException
     * if the value of `millis` is negative
     *
     * @throws  InterruptedException
     * if any thread has interrupted the current thread. The
     * *interrupted status* of the current thread is
     * cleared when this exception is thrown.
     */
    actual fun sleep(millis: Long) {
        try {
            Thread.sleep(millis)
        } catch (e: InterruptedException) {
            e.printStackTrace()
        }
    }

    actual fun isAssignableFrom(a: KClass<*>, b: KClass<*>): Boolean {
        return a.java.isAssignableFrom(b.java)
    }

    actual fun isNumericType(cl: KClass<*>): Boolean {
        return cl != null && (
                isAssignableFrom(Number::class, cl)
                        || cl == Double::class || cl == Int::class || cl == Float::class || cl == Short::class || cl == Long::class
                        || cl == Double::class.javaPrimitiveType || cl == Int::class.javaPrimitiveType || cl == Float::class.javaPrimitiveType || cl == Short::class.javaPrimitiveType || cl == Long::class.javaPrimitiveType
                )    }

    @OptIn(ExperimentalTime::class)
    actual fun isTemporalType(cl: KClass<*>): Boolean {
        return cl == Date::class || cl == java.sql.Date::class || cl == Instant::class
//        return cl != null && (isAssignableFrom(Date::class.java, cl) || isAssignableFrom(Calendar::class.java, cl)
//                || cl == Double::class.javaPrimitiveType || cl == Long::class.javaPrimitiveType)
    }

    actual fun isTemporalValue(value: Any?): Boolean {
        return value is Date || value is java.sql.Date
    }

    actual fun doubleTemporalValue(value: Any?): Double {
        if(value is Date) {
            return value.time.toDouble()
        } else if(value is java.sql.Date) {
            return value.time.toDouble()
        } else {
            throw UnsupportedOperationException()
        }
    }

    actual fun isVisualType(cl: KClass<*>): Boolean {
        return cl.simpleName == "Color"

//        return cl != null && (isAssignableFrom(CPColor::class.java, cl) || isAssignableFrom(
//            CPIcon::class.java,
//            cl
//        ) || isAssignableFrom(
//            Image::class.java, cl
//        ))
    }

    actual fun isArray(cl: KClass<*>): Boolean {
        return cl.java.isArray
    }

    actual val isMac: Boolean by lazy {
        val os = System.getProperty("os.name").lowercase()
        os != null && "mac os x" == os
    }

    /** Current UNIX time in millis */
    actual val currentTimeMillis: Long
        get() = System.currentTimeMillis()
}