package com.macrofocus.common.format

import kotlin.math.absoluteValue
import kotlin.math.round

interface FormatFactoryCommon {
    fun createOrdinalFormat(): CPFormat<Any?> {
        return object : AbstractFormat<Any?>() {
            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value != null && value is Number) {
                    val i = value.toInt()
                    val sufixes = arrayOf("th", "st", "nd", "rd", "th", "th", "th", "th", "th", "th")
                    when ((i.absoluteValue) % 100) {
                        11, 12, 13 -> i.toString() + "th"
                        else -> "" + i + sufixes[i.absoluteValue % 10]
                    }
                } else {
                    null
                }
            }

            override fun parse(text: String?): Any? {
                return null
            }

            override val nativeFormat: Any?
                get() = null
        }
    }

    companion object {
        private val THOUSANDS = 1000L // thou or K

        private val MILLION = 1000000L // m or M

        private val BILLION = 1000000000L // bn or B

        private val TRILLION = 1000000000000L // tn or T
    }

    fun createMilionFormat(decimalFormat: CPFormat<Any?>): CPFormat<Any?> {
        return object : AbstractFormat<Any?>() {
            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value is Number) {
                    val x = value.toDouble()
                    val abs: Double = x.absoluteValue
                    if (abs < THOUSANDS) decimalFormat.format(x) else if (abs < MILLION) decimalFormat.format(x / THOUSANDS)
                        .toString() + "k" else if (abs < BILLION) decimalFormat.format(x / MILLION)
                        .toString() + "m" else if (abs < TRILLION) decimalFormat.format(x / BILLION).toString() + "bn" else decimalFormat.format(x / TRILLION)
                        .toString() + "tn"
                } else {
                    decimalFormat.formatHtml(value, htmlSupported)
                }
            }

            override fun parse(text: String?): Any? {
                return null
            }

            override val nativeFormat: Any?
                get() = null
        }
    }

    fun createDecimalFormat(fractionDigits: Int): CPFormat<Any?>? {
        return createDecimalFormat(fractionDigits, null, null, 1.0, 1.0)
    }

    fun createIntegerFormat(): CPFormat<Any?> {
        return createIntegerFormat(null, null, 1.0)
    }

    fun createPercentDecimalFormat(fractionDigits: Int): CPFormat<Any?> {
        return createPercentDecimalFormat(fractionDigits, 1.0)
    }

    fun createStarFormat(): CPFormat<Any?> {
        return createStarFormat(null)
    }

    fun createStarFormat(na: String?): CPFormat<Any?> {
        return object : AbstractFormat<Any?>() {
            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value != null) {
                    val length: Int
                    length = if (value is Number) {
                        value.toInt()
                    } else {
                        value.toString().length
                    }
                    var text = ""
                    for (i in 0 until length) {
                        text += "*"
                    }
                    text
                } else {
                    na
                }
            }

            override val nativeFormat: Any?
                get() = null

            override fun parse(text: String?): Any? {
                return null
            }
        }
    }

    fun createBooleanFormat(): CPFormat<Any?>? {
        return object : AbstractFormat<Any?>() {
            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value != null) {
                    if (value is Boolean) {
                        if (value) {
                            "True"
                        } else {
                            "False"
                        }
                    } else {
                        null
                    }
                } else {
                    null
                }
            }

            override val nativeFormat: Any?
                get() = null

            override fun parse(text: String?): Any? {
                return null
            }
        }
    }

    fun createFormat(pre: String?, post: String?): CPFormat<Any?>? {
        return object : AbstractFormat<Any?>() {
            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value != null) {
                    var s = value.toString()
                    if (pre != null) {
                        s = pre + s
                    }
                    if (post != null) {
                        s = s + post
                    }
                    s
                } else {
                    ""
                }
            }

            override val nativeFormat: Any?
                get() = null

            override fun parse(text: String?): Any? {
                return null
            }
        }
    }

    fun createDecimalFormat(fractionDigits: Int, pre: String?, post: String?, rounding: Double, factor: Double): CPFormat<Any?> {
        val instance: DecimalFormat = DecimalFormat("")
        instance.maximumFractionDigits = fractionDigits
        return SwingFormat(instance, pre, post, rounding, factor)
    }
    fun createIntegerFormat(pre: String?, post: String?, rounding: Double): CPFormat<Any?>
    fun createPercentDecimalFormat(fractionDigits: Int, rounding: Double): CPFormat<Any?>

    class SwingFormat(
        format: DecimalFormat,
        private val pre: String?,
        private val post: String?,
        private val rounding: Double,
        private val factor: Double
    ) : AbstractFormat<Any?>() {
        private val format: DecimalFormat = format

        override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
            var s: String?
            if (value is Number) {
                var n = value.toDouble()

                if (rounding != 1.0) {
                    n = round(n.toDouble() / rounding) * rounding
                }
                if (factor != 1.0) {
                    n = n * factor
                }
                s = format.format(n)

                if (pre != null) {
                    s = pre + s
                }
                if (post != null) {
                    s = s + post
                }
            } else {
                s = null
            }
            return s
        }

        override fun parse(text: String?): Any? {
            TODO()
//            var text = text
//            return if (text != null) {
//                if (pre != null && !text.startsWith(pre)) {
//                    text = pre + text
//                }
//                if (post != null && !text.endsWith(post)) {
//                    text = text + post
//                }
//                try {
//                    format.parseObject(text)
//                } catch (e: IllegalArgumentException) {
//                    throw CPFormat.ParsingException(e.message, -1)
//                }
//            } else {
//                null
//            }
        }

        override val nativeFormat: DecimalFormat
            get() = format
    }

    private class SwingNumberFormat(pattern: String) : CPFormat<Any?> {
        val numberFormat: DecimalFormat
        override val pattern: String

        override fun format(value: Any?): String {
            return formatHtml(value, false)
        }

        override fun formatHtml(value: Any?, htmlSupported: Boolean): String {
            return if (value != null) {
                val number: Number
                number = if (value is Number) {
                    value
                } else {
                    return value.toString()
                }
                numberFormat.format(number.toDouble())
            } else {
                ""
            }
        }

        override fun parse(text: String?): Any {
            TODO()
        }

        override val nativeFormat: Any
            get() = numberFormat
        override val horizontalAlignment: CPFormat.HorizontalAlignment?
            get() = null

        override fun getClassName(value: Any?): String? {
            return null
        }

        init {
            numberFormat = DecimalFormat(pattern)
            this.pattern = pattern
        }
    }
}