/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.properties

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

abstract class AbstractProperty<T> : Property<T> {
    private val listernersDelegate = lazy { CollectionFactory.copyOnWriteArrayList<PropertyListener<T>>() }
    private val listeners: MutableList<PropertyListener<T>> by listernersDelegate

    override fun addPropertyListener(listener: PropertyListener<T>) {
        listeners.add(listener)
    }

    override fun addWeakPropertyListener(listener: PropertyListener<T>) {
        val weakListener = WeakPropertyListener(listener)
        addPropertyListener(weakListener)
    }

    override fun removePropertyListener(listener: PropertyListener<T>) {
        listeners.remove(listener)
    }

    protected fun notifyPropertyChanged(event: PropertyEvent<T>) {
        if (listernersDelegate.isInitialized()) {
            for (listener in listeners) {
                listener.propertyChanged(event)
            }
        }
    }

    private inner class WeakPropertyListener(listener: PropertyListener<T>) : PropertyListener<T> {
        private val l_ref: WeakReference<PropertyListener<T>> = WeakReference(listener)

        override fun propertyChanged(event: PropertyEvent<T>) {
            val l: PropertyListener<T>? = reference
            if (l != null) {
                l.propertyChanged(event)
            } else {
                removePropertyListener(this as PropertyListener<T>)
            }
        }

        private val reference: PropertyListener<T>?
            get() = l_ref.get()

        override fun toString(): String {
            val l: PropertyListener<T>? = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                l.toString()
            }
        }

    }
}