package com.macrofocus.common.transform

import com.macrofocus.common.interval.BoundedInterval
import com.macrofocus.common.interval.Interval
import com.macrofocus.common.interval.IntervalEvent
import com.macrofocus.common.interval.IntervalListener

/**
 * Default implementation of a OneDScreenTransform.
 */
class ComplexOneDScreenTransform(private var world: BoundedInterval, private val x1: Int, private val x2: Int) :
    AbstractMutableOneDScreenTransform() {
    override var screenMargins = 0
        set(screenMargins) {
            if (field != screenMargins) {
                field = screenMargins
                notifyTransformChanged(ScreenTransformEvent())
            }
        }
    override var screenSize: Int = 0
        set(screenSize) {
            if (field != screenSize) {
                field = screenSize
                notifyTransformChanged(ScreenTransformEvent())
            }
        }
    private var invertDeviceCoordinates = false
    override var isWorldCoordinatesInverted: Boolean = false
        private set

    private val intervalListener: IntervalListener = object : IntervalListener {
        override fun intervalChanged(event: IntervalEvent) {
            notifyTransformChanged(ScreenTransformEvent())
        }
    }

    init {
        this.screenSize = x2 - x1

        world.addWeakIntervalListener(intervalListener)
        this.screenSize = screenSize
    }


    constructor(world: BoundedInterval, screenSize: Int) : this(world, 0, screenSize)

    constructor(
        world: BoundedInterval,
        screenSize: Int,
        invertDeviceCoordinates: Boolean,
        invertWorldCoordinates: Boolean
    ) : this(world, screenSize) {
        this.invertDeviceCoordinates = invertDeviceCoordinates
        this.isWorldCoordinatesInverted = invertWorldCoordinates
    }

    constructor(
        world: BoundedInterval,
        x1: Int,
        x2: Int,
        invertDeviceCoordinates: Boolean,
        invertWorldCoordinates: Boolean
    ) : this(world, x1, x2) {
        this.invertDeviceCoordinates = invertDeviceCoordinates
        this.isWorldCoordinatesInverted = invertWorldCoordinates
    }

    override fun screenToWorld(screen: Int): Double {
        var normalized: Double

        // convert screen to viewport (turn upside down if device coords should be inverted)
        normalized = ((screen - x1) - (screenMargins / 2)).toDouble() / (screenSize - screenMargins).toDouble()
        if (invertDeviceCoordinates) normalized = 1.0 - normalized

        if (isWorldCoordinatesInverted) {
            normalized = 1.0 - normalized
        }

        // simple de-normalization
        normalized = (normalized * worldRange) + worldMin

        return normalized
    }

    override fun worldToScreen(world: Double): Int {
        return worldToScreenPrecise(world).toInt()
    }

    override fun worldToScreenPrecise(world: Double): Double {
        return if (isWorldCoordinatesInverted xor invertDeviceCoordinates) {
            // Inverted
            ((worldMax - world) * (((screenSize - screenMargins)).toDouble()) / worldRange) + (screenMargins / 2.0) + x1
        } else {
            ((world - worldMin) * (((screenSize - screenMargins)).toDouble()) / worldRange) + (screenMargins / 2.0) + x1
        }
    }

    override val worldInterval: Interval
        get() = world

    override fun setWorldInterval(world: BoundedInterval) {
        this.world.removeIntervalListener(intervalListener)
        this.world = world
        this.world.addIntervalListener(intervalListener)
        notifyTransformChanged(ScreenTransformEvent())
    }

    override val isAffine: Boolean
        get() = true

    override val worldMin: Double
        get() {
            return if (isWorldCoordinatesInverted) {
                world.minimum + (world.maximum - (world.start + world.extent))
            } else {
                world.start
            }
        }

    override val worldMax: Double
        get() {
            return if (isWorldCoordinatesInverted) {
                world.minimum + (world.maximum - (world.start))
            } else {
                world.start + world.extent
            }
        }

    override val worldRange: Double
        get() = world.extent

    override fun toString(): String {
        return "ComplexOneDScreenTransform{" +
                "world=" + world +
                ", screenMargins=" + screenMargins +
                ", x1=" + x1 +
                ", x2=" + x2 +
                ", screenSize=" + screenSize +
                ", invertDeviceCoordinates=" + invertDeviceCoordinates +
                ", invertWorldCoordinates=" + isWorldCoordinatesInverted +
                '}'
    }
}
