package com.macrofocus.order

import com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList
import com.macrofocus.common.collection.WeakReference

abstract class AbstractOrder<E> : Order<E> {
    private val listeners: MutableList<OrderListener<E>>
    override fun addOrderListener(listener: OrderListener<E>) {
        listeners.add(listener) 
    }

    override fun addWeakOrderListener(listener: OrderListener<E>) {
        listeners.add(WeakOrderListener(listener)) 
    }

    override fun removeOrderListener(listener: OrderListener<E>) {
        if (listener is WeakOrderListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: OrderListener<E>? = null
            for (orderListener in listeners) {
                var comparable: OrderListener<E>?
                comparable = if (orderListener is WeakOrderListener) {
                    orderListener.reference
                } else {
                    orderListener
                }
                if (listener == comparable) {
                    toRemove = orderListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    fun removeFilterListeners() {
        listeners.clear() 
    }

    protected fun notifyOrderChanged(event: OrderEvent<E>?) {
        for (listener in listeners) {
            listener.orderChanged(event)
        }
    }

    protected fun notifyOrderVisibility(event: OrderEvent<E>) {
        for (listener in listeners) {
            listener.orderVisibility(event)
        }
    }

    protected fun notifyOrderAdded(event: OrderEvent<E>) {
        for (listener in listeners) {
            listener.orderAdded(event)
        }
    }

    protected fun notifyOrderRemoved(event: OrderEvent<E>) {
        for (listener in listeners) {
            listener.orderRemoved(event)
        }
    }

    private inner class WeakOrderListener(listener: OrderListener<E>) : OrderListener<E> {
        private val l_ref: WeakReference<OrderListener<E>>
        override fun orderChanged(event: OrderEvent<E>?) {
            val l: OrderListener<E>? = reference
            if (l != null) {
                l.orderChanged(event)
            } else {
                removeOrderListener(this as OrderListener<E>)
            }
        }

        override fun orderVisibility(event: OrderEvent<E>) {
            val l: OrderListener<E>? = reference
            if (l != null) {
                l.orderVisibility(event)
            } else {
                removeOrderListener(this as OrderListener<E>)
            }
        }

        override fun orderAdded(event: OrderEvent<E>) {
            val l: OrderListener<E>? = reference
            if (l != null) {
                l.orderAdded(event)
            } else {
                removeOrderListener(this as OrderListener<E>)
            }
        }

        override fun orderRemoved(event: OrderEvent<E>) {
            val l: OrderListener<E>? = reference
            if (l != null) {
                l.orderRemoved(event)
            } else {
                removeOrderListener(this as OrderListener<E>)
            }
        }

        val reference: OrderListener<E>?
            get() = l_ref.get()

        init {
            l_ref = WeakReference<OrderListener<E>>(listener)
        }
    }

    init {
        listeners = copyOnWriteArrayList<OrderListener<E>>()
    }
}