/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.collection

import java.util.concurrent.ConcurrentHashMap

actual class ConcurrentHashSet<K> : MutableSet<K> {
    private val map: ConcurrentHashMap<K, K> = ConcurrentHashMap<K, K>()

    actual override fun add(element: K): Boolean {
        val contains = map.containsKey(element)
        map[element] = element
        return !contains
    }

    actual override fun addAll(elements: Collection<K>): Boolean {
        var changed = false
        for (element in elements) {
            changed = add(element) || changed
        }
        return changed
    }

    actual override fun clear() {
        map.clear()
    }

    actual override fun contains(element: K): Boolean {
        return map.containsKey(element)
    }

    actual override fun containsAll(elements: Collection<K>): Boolean {
        for (o in elements) {
            if (!contains(o)) return false
        }
        return true
    }

    actual override fun isEmpty(): Boolean {
        return map.size == 0
    }

    actual override fun iterator(): MutableIterator<K> {
        return map.values.iterator()
    }

    actual override fun remove(element: K): Boolean {
        return map.remove(element) != null
    }

    actual override fun removeAll(elements: Collection<K>): Boolean {
        var changed = false
        for (o in elements) {
            changed = remove(o) || changed
        }
        return changed
    }

    actual override fun retainAll(elements: Collection<K>): Boolean {
        clear()
        for (element in elements) {
            add(element)
        }
        return true
    }

    actual override val size: Int
        get() = map.size

    fun toArray(): Array<Any?>? {
        return map.values.toTypedArray()
    }
}

actual fun <K> emptyConcurrentHashSet(): ConcurrentHashSet<K> {
   return ConcurrentHashSet()
}