/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.hierarchy

import com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList
import com.macrofocus.common.collection.WeakReference
import kotlin.jvm.Transient

/**
 * This class provides a skeletal implementation of the Hierarchy interface to minimize the effort required to implement
 * this interface.
 */
abstract class AbstractHierarchy<T> : Hierarchy<T> {
    @Transient
    override var listeners: MutableList<HierarchyListener<T>>? = null
    private var notifyListeners = true
    override fun addHierarchyListener(listener: HierarchyListener<T>) {
        if (listeners == null) {
            listeners = copyOnWriteArrayList<HierarchyListener<T>>()
        }
        listeners!!.add(listener)
    }

    override fun addWeakHierarchyListener(listener: HierarchyListener<T>) {
        if (listeners == null) {
            listeners = copyOnWriteArrayList<HierarchyListener<T>>()
        }
        val weakListener: WeakHierarchyListener = WeakHierarchyListener(listener)
        listeners!!.add(listener)
    }

    override fun removeHierarchyListener(hierarchyListener: HierarchyListener<T>) {
        if (listeners != null) {
            listeners!!.remove(hierarchyListener)
            if (listeners!!.isEmpty()) {
                listeners = null
            }
        }
    }

    override fun removeHierarchyListeners() {
        if (listeners != null) {
            listeners!!.clear()
            if (listeners!!.isEmpty()) {
                listeners = null
            }
        }
    }

    override fun setNotifyListeners(enable: Boolean) {
        if (enable.also { notifyListeners = it }) {
            notifyListeners = enable
            if (enable) {
                notifyHierarchyStructureChanged()
            }
        }
    }

    override fun isRoot(node: T): Boolean {
        return root === node
    }

    /**
     * Returns the next sibling of this node in the parent's children array.
     * Returns null if this node has no parent or is the parent's last child.
     * This method performs a linear search that is O(n) where n is the number
     * of children; to traverse the entire array, use the parent's child
     * enumeration instead.
     *
     * @return the sibling of this node that immediately follows this node
     */
    fun getNextSibling(parent: T?, child: T): T? {
        val retval: T?
        retval = parent?.let { getChildAfter(it, child) }
        return retval
    }

    /**
     * Returns true if `anotherNode` is a sibling of (has the
     * same parent as) this node.  A node is its own sibling.  If
     * `anotherNode` is null, returns false.
     *
     * @param anotherNode node to test as sibling of this node
     * @return true if `anotherNode` is a sibling of this node
     */
    fun isNodeSibling(aNode: T, anotherNode: T?): Boolean {
        val retval: Boolean
        retval = if (anotherNode == null) {
            false
        } else if (anotherNode === this) {
            true
        } else {
            val myParent: T? = getParent(aNode)
            myParent != null && myParent === getParent(anotherNode)
        }
        return retval
    }

    /**
     * Returns the previous sibling of this node in the parent's children
     * array.  Returns null if this node has no parent or is the parent's
     * first child.  This method performs a linear search that is O(n) where n
     * is the number of children.
     *
     * @return the sibling of this node that immediately precedes this node
     */
    fun getPreviousSibling(parent: T?, node: T): T? {
        val retval: T?
        retval = parent?.let { getChildBefore(it, node) }
        return retval
    }

    /**
     * Returns the child in this node's child array that immediately
     * follows `aChild`, which must be a child of this node.  If
     * `aChild` is the last child, returns null.  This method
     * performs a linear search of this node's children for
     * `aChild` and is O(n) where n is the number of children; to
     * traverse the entire array of children, use an enumeration instead.
     *
     * @return the child of this node that immediately follows
     * `aChild`
     * @throws IllegalArgumentException if `aChild` is
     * null or is not a child of this node
     */
    fun getChildAfter(parent: T, child: T?): T? {
        requireNotNull(child) { "argument is null" }
        val index = getIndexOfChild(parent, child) // linear search
        require(index != -1) { "node is not a child" }
        return if (index < getChildCount(parent) - 1) {
            getChild(parent, index + 1)
        } else {
            null
        }
    }

    /**
     * Returns the child in this node's child array that immediately
     * precedes `aChild`, which must be a child of this node.  If
     * `aChild` is the first child, returns null.  This method
     * performs a linear search of this node's children for `aChild`
     * and is O(n) where n is the number of children.
     *
     * @return the child of this node that immediately precedes
     * `aChild`
     * @throws IllegalArgumentException if `aChild` is null
     * or is not a child of this node
     */
    fun getChildBefore(parent: T, child: T?): T? {
        requireNotNull(child) { "argument is null" }
        val index = getIndexOfChild(parent, child) // linear search
        require(index != -1) { "argument is not a child" }
        return if (index > 0) {
            getChild(parent, index - 1)
        } else {
            null
        }
    }

    /**
     * Returns the depth of the tree rooted at this node -- the longest
     * distance from this node to a leaf.  If this node has no children,
     * returns 0.  This operation is much more expensive than
     * `getLevel()` because it must effectively traverse the entire
     * tree rooted at this node.
     *
     * @see .getLevel
     *
     * @return  the depth of the tree whose root is this node
     */
    override val depth: Int
        get() {
            var last: T? = null
            val it = breadthFirstIterator()
            for (t in it) {
                last = t
            }
            if (last == null) {
                throw Error("nodes should be null")
            }
            return getLevel(last) - getLevel(root)
        }

    /**
     * Returns the number of levels above this node -- the distance from
     * the root to this node.  If this node is the root, returns 0.
     *
     * @see .getDepth
     *
     * @return  the number of levels above this node
     */
    override fun getLevel(node: T): Int {
        var ancestor: T
        var levels = 0
        ancestor = node
        while (getParent(ancestor).also { ancestor = it!! } != null) {
            levels++
        }
        return levels
    }

    /**
     * Returns the path from the root, to get to this node.  The last
     * element in the path is this node.
     *
     * @return an array of TreeNode objects giving the path, where the
     * first element in the path is the root and the last
     * element is this node.
     */
    override fun getPath(node: T): List<T>? {
        val path = getPathToRoot(node, 0)
        return if (path != null) {
            val list: MutableList<T> = ArrayList<T>(path.size)
            for (i in path.indices) {
                val o = path[i]
                list.add(o as T)
            }
            list
        } else {
            null
        }
    }

    /**
     * Indicates whether an element is ancestor of another one.
     *
     * @param ancestor   potential ancestor
     * @param descendant potential descendant
     * @return `true` if `ancestor` is equal to `descendant` or if `ancestor`
     * is the parent of `descendant` or if `ancestor` is the parent of an ancestor of
     * `descendant`; returns `false` otherwise
     */
    fun isAncestor(ancestor: T, descendant: T): Boolean {
        var descendant: T? = descendant
        while (descendant != null) {
            if (descendant === ancestor) return true
            descendant = getParent(descendant)
        }
        return false
    }

    protected fun notifyHierarchyNodeInserted(child: T, parent: T, index: Int, isAdjusting: Boolean) {
        if (listeners != null && notifyListeners) {
            val event = HierarchyEvent(this, HierarchyEvent.Type.Inserted, child, parent, index, isAdjusting)
            for (listener in listeners!!) {
                listener.hierarchyNodeInserted(event)
            }
        }
    }

    override fun notifyHierarchyNodeChanged(child: T, parent: T, index: Int, isAdjusting: Boolean) {
        if (listeners != null && notifyListeners) {
            val event = HierarchyEvent(this, HierarchyEvent.Type.Changed, child, parent, index, isAdjusting)
            for (listener in listeners!!) {
                listener.hierarchyNodeChanged(event)
            }
        }
    }

    protected fun notifyHierarchyNodeRemoved(child: T, parent: T, index: Int, isAdjusting: Boolean) {
        if (listeners != null && notifyListeners) {
            val event = HierarchyEvent(this, HierarchyEvent.Type.Removed, child, parent, index, isAdjusting)
            for (listener in listeners!!) {
                listener.hierarchyNodeRemoved(event)
            }
        }
    }

    protected fun notifyHierarchyStructureChanged() {
        if (listeners != null && notifyListeners) {
            val event = HierarchyEvent<T>(this, HierarchyEvent.Type.StructureChanged, null, null, -1, false)
            for (listener in listeners!!) {
                listener.hierarchyStructureChanged(event)
            }
        }
    }

    override fun preorderIterator(parent: T): Iterable<T> {
        return object : Iterable<T> {
            override fun iterator(): Iterator<T> {
                return PreorderEnumeration<T>(this@AbstractHierarchy, parent)
            }
        }
    }

    override fun breadthFirstIterator(parent: T): Iterable<T> {
        return object : Iterable<T> {
            override fun iterator(): Iterator<T> {
                return com.macrofocus.hierarchy.BreadthFirstIterator<T>(this@AbstractHierarchy, parent)
            }
        }
    }

    override fun depthFirstIterator(parent: T): Iterable<T> {
        return object : Iterable<T> {
            override fun iterator(): Iterator<T> {
                return PostorderEnumeration<T>(this@AbstractHierarchy, parent)
            }
        }
    }

    override fun leavesIterator(parent: T): Iterable<T> {
        return object : Iterable<T> {
            override fun iterator(): Iterator<T> {
                return LeavesEnumeration<T>(this@AbstractHierarchy, parent)
            }
        }
    }

    override fun preorderIterator(): Iterable<T> {
        return preorderIterator(root)
    }

    override fun breadthFirstIterator(): Iterable<T> {
        return breadthFirstIterator(root)
    }

    override fun depthFirstIterator(): Iterable<T> {
        return depthFirstIterator(root)
    }

    override fun leavesIterator(): Iterable<T> {
        return leavesIterator(root)
    }

    /**
     * Builds the parents of node up to and including the root node,
     * where the original node is the last element in the returned array.
     * The length of the returned array gives the node's depth in the
     * tree.
     *
     * @param aNode the TreeNode to get the path for
     */
    override fun getPathToRoot(aNode: T): Array<Any?>? {
        return getPathToRoot(aNode, 0)
    }
    //
    //  Child Queries
    //
    /**
     * Returns true if `aNode` is a child of this node.  If
     * `aNode` is null, this method returns false.
     *
     * @return  true if `aNode` is a child of this node; false if
     * `aNode` is null
     */
    fun isNodeChild(parent: T, aNode: T?): Boolean {
        val retval: Boolean
        retval = if (aNode == null) {
            false
        } else {
            if (getChildCount(parent) == 0) {
                false
            } else {
                getParent(aNode) === parent
            }
        }
        return retval
    }

    /**
     * Returns this node's first child.  If this node has no children,
     * throws NoSuchElementException.
     *
     * @return  the first child of this node
     * @exception java.util.NoSuchElementException  if this node has no children
     */
    fun getFirstChild(node: T): T {
        if (getChildCount(node) == 0) {
            throw NoSuchElementException("node has no children")
        }
        return getChild(node, 0)
    }

    /**
     * Returns this node's last child.  If this node has no children,
     * throws NoSuchElementException.
     *
     * @return  the last child of this node
     * @exception       NoSuchElementException  if this node has no children
     */
    fun getLastChild(node: T): T {
        if (getChildCount(node) == 0) {
            throw NoSuchElementException("node has no children")
        }
        return getChild(node, getChildCount(node) - 1)
    }

    /**
     * Builds the parents of node up to and including the root node,
     * where the original node is the last element in the returned array.
     * The length of the returned array gives the node's depth in the
     * tree.
     *
     * @param aNode the TreeNode to get the path for
     * @param depth an int giving the number of steps already taken towards
     * the root (on recursive calls), used to size the returned array
     * @return an array of TreeNodes giving the path from the root to the
     * specified node
     */
    protected fun getPathToRoot(aNode: T?, depth: Int): Array<Any?>? {
        var depth = depth
        val retNodes: Array<Any?>?
        // This method recurses, traversing towards the root in order
        // size the array. On the way back, it fills in the nodes,
        // starting from the root and working back to the original node.

        /* Check for null, in case someone passed in a null node, or
           they passed in an element that isn't rooted at root. */if (aNode == null) {
            retNodes = if (depth == 0) return null else arrayOfNulls(depth)
        } else {
            depth++
            retNodes = if (aNode === root) arrayOfNulls(depth) else getPathToRoot(getParent(aNode), depth)
            retNodes!![retNodes.size - depth] = aNode
        }
        return retNodes
    }

    /**
     * Returns true if this node has no children.  To distinguish between
     * nodes that have no children and nodes that *cannot* have
     * children (e.g. to distinguish files from empty directories), use this
     * method in conjunction with `getAllowsChildren`
     *
     * @return    true if this node has no children
     */
    override fun isLeaf(node: T): Boolean {
        return getChildCount(node) == 0
    }

    /**
     * Finds and returns the first leaf that is a descendant of this node --
     * either this node or its first child's first leaf.
     * Returns this node if it is a leaf.
     *
     * @see .isLeaf
     *
     * @return  the first leaf in the subtree rooted at this node
     */
    override fun getFirstLeaf(node: T): T {
        var node = node
        while (!isLeaf(node)) {
            node = getFirstChild(node)
        }
        return node
    }

    /**
     * Finds and returns the last leaf that is a descendant of this node --
     * either this node or its last child's last leaf.
     * Returns this node if it is a leaf.
     *
     * @see .isLeaf
     *
     * @return  the last leaf in the subtree rooted at this node
     */
    override fun getLastLeaf(node: T): T {
        var node = node
        while (!isLeaf(node)) {
            node = getLastChild(node)
        }
        return node
    }

    /**
     * Returns the leaf after this node or null if this node is the
     * last leaf in the tree.
     *
     *
     * In this implementation of the `MutableNode` interface,
     * this operation is very inefficient. In order to determine the
     * next node, this method first performs a linear search in the
     * parent's child-list in order to find the current node.
     *
     *
     * That implementation makes the operation suitable for short
     * traversals from a known position. But to traverse all of the
     * leaves in the tree, you should use `depthFirstEnumeration`
     * to enumerate the nodes in the tree and use `isLeaf`
     * on each node to determine which are leaves.
     *
     * @see .isLeaf
     *
     * @return  returns the next leaf past this node
     */
    override fun getNextLeaf(node: T): T? {
        val nextSibling: T?
        val myParent: T = getParent(node) ?: return null
        nextSibling = getNextSibling(myParent, node) // linear search
        return nextSibling?.let { getFirstLeaf(it) } ?: getNextLeaf(myParent)
        // tail recursion
    }

    /**
     * Returns the leaf before this node or null if this node is the
     * first leaf in the tree.
     *
     *
     * In this implementation of the `MutableNode` interface,
     * this operation is very inefficient. In order to determine the
     * previous node, this method first performs a linear search in the
     * parent's child-list in order to find the current node.
     *
     *
     * That implementation makes the operation suitable for short
     * traversals from a known position. But to traverse all of the
     * leaves in the tree, you should use `depthFirstEnumeration`
     * to enumerate the nodes in the tree and use `isLeaf`
     * on each node to determine which are leaves.
     *
     * @see .isLeaf
     *
     * @return  returns the leaf before this node
     */
    override fun getPreviousLeaf(node: T): T? {
        val previousSibling: T?
        val myParent: T = getParent(node) ?: return null
        previousSibling = getPreviousSibling(myParent, node) // linear search
        return previousSibling?.let { getLastLeaf(it) } ?: getPreviousLeaf(myParent)
        // tail recursion
    }

    /**
     * Returns the total number of leaves that are descendants of this node.
     * If this node is a leaf, returns `1`.  This method is O(n)
     * where n is the number of descendants of this node.
     *
     * @return  the number of leaves beneath this node
     */
    override fun getLeafCount(node: T): Int {
        var count = 0
        val it = breadthFirstIterator(node) // order matters not
        for (t in it) {
            if (isLeaf(t)) {
                count++
            }
        }
        if (count < 1) {
            throw Error("hierarchy has zero leaves")
        }
        return count
    }

    override fun toString(): String {
        val count: Int
        count = if (listeners != null) {
            listeners!!.size
        } else {
            0
        }
        return this::class.simpleName + "[listeners=" + count + "]"
    }

    private inner class WeakHierarchyListener(listener: HierarchyListener<T>) : HierarchyListener<T> {
        private val l_ref: WeakReference<HierarchyListener<T>>
        override fun hierarchyNodeInserted(event: HierarchyEvent<T>) {
            val l = reference
            if (l != null) {
                l.hierarchyNodeInserted(event)
            } else {
                removeHierarchyListener(this as HierarchyListener<T>)
            }
        }

        override fun hierarchyNodeChanged(event: HierarchyEvent<T>) {
            val l = reference
            if (l != null) {
                l.hierarchyNodeChanged(event)
            } else {
                removeHierarchyListener(this as HierarchyListener<T>)
            }
        }

        override fun hierarchyNodeRemoved(event: HierarchyEvent<T>) {
            val l = reference
            if (l != null) {
                l.hierarchyNodeRemoved(event)
            } else {
                removeHierarchyListener(this as HierarchyListener<T>)
            }
        }

        override fun hierarchyStructureChanged(event: HierarchyEvent<T>) {
            val l = reference
            if (l != null) {
                l.hierarchyStructureChanged(event)
            } else {
                removeHierarchyListener(this as HierarchyListener<T>)
            }
        }

        private val reference: HierarchyListener<T>?
            private get() = l_ref.get()

        override fun toString(): String {
            val l = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                super.toString()
            }
        }

        init {
            l_ref = WeakReference<HierarchyListener<T>>(listener)
        }
    }
}