/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.properties

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

abstract class AbstractProperties<K> : Properties<K> {
    private val listeners: MutableList<PropertiesListener<K>> = CollectionFactory.copyOnWriteArrayList()

    override fun addPropertiesListener(listener: PropertiesListener<K>) {
        listeners.add(listener)
    }

    override fun addWeakPropertiesListener(listener: PropertiesListener<K>) {
        val weakListener: WeakPropertiesListener = WeakPropertiesListener(listener)
        listeners.add(weakListener)
    }

    override fun removePropertiesListener(listener: PropertiesListener<K>) {
        if (listener is WeakPropertiesListener) {
            val removed = listeners.remove(listener)
//                assert(removed) { listener }
        } else {
            var toRemove: PropertiesListener<K>? = null
            for (propertiesListener in listeners) {
                var comparable: PropertiesListener<K>?
                if (propertiesListener is WeakPropertiesListener) {
                    comparable = propertiesListener.reference
                } else {
                    comparable = propertiesListener
                }
                if (listener.equals(comparable)) {
                    toRemove = propertiesListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
//                    assert(removed) { listener }
            }
        }
        listeners.remove(listener)
    }

    protected fun notifyPropertyChanged(name: K, event: PropertyEvent<Any?>) {
        for (listener in listeners) {
            listener.propertyChanged(name, event)
        }
    }

    private inner class WeakPropertiesListener(listener: PropertiesListener<K>) : PropertiesListener<K> {
        private val l_ref: WeakReference<PropertiesListener<K>> = WeakReference(listener)

        override fun propertyChanged(name: K, event: PropertyEvent<Any?>) {
            val l: PropertiesListener<K>? = reference
            if (l != null) {
                l.propertyChanged(name, event)
            } else {
                removePropertiesListener(this)
            }
        }

        val reference: PropertiesListener<K>?
            public get() = l_ref.get()

        override fun toString(): String {
            val l: PropertiesListener<K>? = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                l.toString()
            }
        }
    }
}