/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.properties

import com.macrofocus.common.collection.EnumMap

/**
 * Default data model for a collection of properties.
 */
class EnumProperties<K : Enum<K>> (enums : Array<K>): AbstractProperties<K>(), MutableProperties<K> {
    private val propertyMap: EnumMap<K, ExposedProperty<Any?>> = EnumMap(enums)
    private val listenerMap: EnumMap<K, PropertyListener<Any?>> = EnumMap<K, PropertyListener<Any?>>(enums)

   constructor(enums : Array<K>, vararg pairs: Pair<K,Any?>) : this(enums) {
       for (pair in pairs) {
           createProperty(pair.first, pair.second)
       }
   }

    override operator fun iterator(): MutableIterator<K> {
        return propertyMap.keys.iterator()
    }

    override fun <T> createProperty(name: K, value: T): MutableProperty<T> {
        val property: MutableProperty<T> = SimpleProperty.newInstance(value)
        return addProperty<T>(name, property)
    }

    override fun <T> addProperty(name: K, property: MutableProperty<T>): MutableProperty<T> {
//        assert(!propertyMap!!.containsKey(name)) { "Property $name already exists" }
        val exposedProperty: ExposedProperty<Any?> = ExposedProperty<Any?>(property as MutableProperty<Any?>)
        propertyMap[name] = exposedProperty
        val listener = NamedPropertyListener(name)
        exposedProperty.addPropertyListener(listener)
        listenerMap[name] = listener
        return exposedProperty as MutableProperty<T>
    }

    override fun <T> replaceProperty(name: K, property: MutableProperty<T>): MutableProperty<T> {
//        assert(propertyMap!!.containsKey(name))
        propertyMap[name]!!.setSecretProperty(property as MutableProperty<Any?>)
        return property
    }

    override fun getValue(name: K): Any? {
        return propertyMap[name]!!.value
    }

    override fun setValue(name: K, value: Any?) {
        propertyMap[name]!!.value = value
    }

    override fun getProperty(name: K): MutableProperty<Any?> {
        return propertyMap[name]!!
    }

    private inner class NamedPropertyListener(private val name: K) : PropertyListener<Any?> {
        override fun propertyChanged(event: PropertyEvent<Any?>) {
            notifyPropertyChanged(name, event)
        }

    }

    private inner class ExposedProperty<T>(secret: MutableProperty<T>) : AbstractMutableProperty<T>(), MutableProperty<T> {
        private var secret: MutableProperty<T>
        
        var listener: PropertyListener<T> = object : PropertyListener<T> {
            override fun propertyChanged(event: PropertyEvent<T>) {
                notifyPropertyChanged(event)
            }
        }

        fun setSecretProperty(secret: MutableProperty<T>) {
            if (this.secret !== secret) {
                this.secret.removePropertyListener(listener)
                val oldValue: T = this.secret.value
                val newValue: T = secret.value
                this.secret = secret
                this.secret.addPropertyListener(listener)
                if (oldValue !== newValue) {
                    notifyPropertyChanged(PropertyEvent(oldValue, newValue))
                }
            }
        }

        override var value: T
            get() = secret.value
            set(value) {secret.value = value}

        init {
            this.secret = secret
            this.secret.addPropertyListener(listener)
        }
    }
}