/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import com.macrofocus.common.math.isInfinite
import com.macrofocus.common.math.isNaN
import kotlin.math.abs

actual interface JsonNumber : JsonValue {
    actual val number: Double
}
/**
 * Server-side implementation of JsonNumber.
 */
class JvmJsonNumber(override val number: Double) : JsonNumber {
    override fun asBoolean(): Boolean {
        return if (isNaN(number) || abs(number) == 0.0) false else true
    }

    override fun asNumber(): Double {
        return number
    }

    override fun asString(): String {
        return toJson()
    }

    override fun asJsonArray(): JsonArray? {
        return null
    }

    override fun asJsonObject(): JsonObject? {
        return null
    }

    override val `object`: Any
        get() = number
    override val type: JsonType
        get() = JsonType.NUMBER

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JsonValue).`object`
    }

    override fun toNative(): Any? {
        return this
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        visitor.visit(number, ctx)
    }

    override fun toJson(): String {
        if (isInfinite(number) || isNaN(number)) {
            return "null"
        }
        var toReturn = number.toString()
        if (toReturn.endsWith(".0")) {
            toReturn = toReturn.substring(0, toReturn.length - 2)
        }
        return toReturn
    }
}