/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import com.macrofocus.common.math.isInfinite
import com.macrofocus.common.math.isNaN
import kotlin.experimental.ExperimentalNativeApi

/**
 * Direct port of json2.js at http://www.json.org/json2.js to GWT.
 */
object JsonUtil {
    /**
     * Convert special control characters into unicode escape format.
     */
    fun escapeControlChars(text: String): String {
        val toReturn = StringBuilder()
        for (i in 0 until text.length) {
            val c = text[i]
            if (isControlChar(c)) {
                toReturn.append(escapeCharAsUnicode(c))
            } else {
                toReturn.append(c)
            }
        }
        return toReturn.toString()
    }

    @Throws(JsonException::class)
    fun <T : JsonValue?> parse(json: String): T {
        return JsonFactory.parse(json)
    }

    /**
     * Safely escape an arbitrary string as a JSON string literal.
     */
    fun quote(value: String?): String {
        val toReturn = StringBuilder("\"")
        for (i in 0 until value!!.length) {
            val c = value[i]
            var toAppend = c.toString()
            when (c) {
                '\b' -> toAppend = "\\b"
                '\t' -> toAppend = "\\t"
                '\n' -> toAppend = "\\n"
//                '\f' -> toAppend = "\\f" // ToDo: Illegal!
                '\r' -> toAppend = "\\r"
                '"' -> toAppend = "\\\""
                '\\' -> toAppend = "\\\\"
                else -> if (isControlChar(c)) {
                    toAppend = escapeCharAsUnicode(c)
                }
            }
            toReturn.append(toAppend)
        }
        toReturn.append("\"")
        return toReturn.toString()
    }

    /**
     * Converts a Json Object to Json format.
     *
     * @param jsonValue  json object to stringify
     * @return json formatted string
     */
    fun stringify(jsonValue: JsonValue): String {
        return stringify(jsonValue, 0)
    }

    /**
     * Converts a JSO to Json format.
     *
     * @param jsonValue    json object to stringify
     * @param spaces number of spaces to indent in pretty print mode
     * @return json formatted string
     */
    fun stringify(jsonValue: JsonValue, spaces: Int): String {
        val sb = StringBuilder()
        for (i in 0 until spaces) {
            sb.append(' ')
        }
        return stringify(jsonValue, sb.toString())
    }

    /**
     * Converts a Json object to Json formatted String.
     *
     * @param jsonValue    json object to stringify
     * @param indent optional indention prefix for pretty printing
     * @return json formatted string
     */
    fun stringify(jsonValue: JsonValue, indent: String?): String {
        val sb = StringBuilder()
        val isPretty = indent != null && "" != indent
        StringifyJsonVisitor(indent, sb, isPretty).accept(jsonValue)
        return sb.toString()
    }

    /**
     * Turn a single unicode character into a 32-bit unicode hex literal.
     */
    private fun escapeCharAsUnicode(toEscape: Char): String {
        val hexValue: String = toEscape.code.toString(16)
        val padding = 4 - hexValue.length
        return "\\u" + "0000".substring(0, padding) + hexValue
    }

    private fun isControlChar(c: Char): Boolean {
        return (c.toInt() >= 0x00 && c.toInt() <= 0x1f
                || c.toInt() >= 0x7f && c.toInt() <= 0x9f
                || c == '\u00ad' || c == '\u070f' || c == '\u17b4' || c == '\u17b5' || c == '\ufeff' || c >= '\u0600' && c <= '\u0604'
                || c >= '\u200c' && c <= '\u200f'
                || c >= '\u2028' && c <= '\u202f'
                || c >= '\u2060' && c <= '\u206f'
                || c >= '\ufff0' && c <= '\uffff')
    }

    private class StringifyJsonVisitor(
        private val indent: String?, private val sb: StringBuilder,
        private val pretty: Boolean
    ) : JsonVisitor() {
        companion object {
            private var skipKeys: Set<String>? = null

            init {
                val toSkip: MutableSet<String> = HashSet()
                toSkip.add("\$H")
                toSkip.add("__gwt_ObjectId")
                skipKeys = toSkip
            }
        }

        private var indentLevel: String = ""
        private val visited: MutableSet<JsonValue>
        override fun endVisit(array: JsonArray, ctx: JsonContext) {
            if (pretty) {
                indentLevel = indentLevel
                    .substring(0, indentLevel.length - indent!!.length)
                sb.append('\n')
                sb.append(indentLevel)
            }
            sb.append("]")
            visited.remove(array)
        }

        @OptIn(ExperimentalNativeApi::class)
        override fun endVisit(`object`: JsonObject, ctx: JsonContext) {
            if (pretty) {
                indentLevel = indentLevel
                    .substring(0, indentLevel.length - indent!!.length)
                sb.append('\n')
                sb.append(indentLevel)
            }
            sb.append("}")
            visited.remove(`object`)
            assert(!visited.contains(`object`))
        }

        override fun visit(number: Double, ctx: JsonContext) {
            sb.append(
                if (isInfinite(number) || isNaN(number)) "null" else format(
                    number
                )
            )
        }

        override fun visit(string: String?, ctx: JsonContext) {
            sb.append(quote(string))
        }

        override fun visit(bool: Boolean, ctx: JsonContext) {
            sb.append(bool)
        }

        override fun visit(array: JsonArray, ctx: JsonContext): Boolean {
            checkCycle(array)
            sb.append("[")
            if (pretty) {
                sb.append('\n')
                indentLevel += indent
                sb.append(indentLevel)
            }
            return true
        }

        override fun visit(`object`: JsonObject, ctx: JsonContext): Boolean {
            checkCycle(`object`)
            sb.append("{")
            if (pretty) {
                sb.append('\n')
                indentLevel += indent
                sb.append(indentLevel)
            }
            return true
        }

        override fun visitIndex(index: Int, ctx: JsonContext): Boolean {
            commaIfNotFirst(ctx)
            return true
        }

        override fun visitKey(key: String?, ctx: JsonContext): Boolean {
            if ("" == key) {
                return true
            }
            // skip properties injected by GWT runtime on JSOs
            if (skipKeys!!.contains(key)) {
                return false
            }
            commaIfNotFirst(ctx)
            sb.append(quote(key) + ":")
            if (pretty) {
                sb.append(' ')
            }
            return true
        }

        override fun visitNull(ctx: JsonContext) {
            sb.append("null")
        }

        private fun checkCycle(value: JsonValue) {
            if (visited.contains(value)) {
                throw JsonException("Cycled detected during stringify")
            } else {
                visited.add(value)
            }
        }

        private fun commaIfNotFirst(ctx: JsonContext) {
            if (!ctx.isFirst) {
                sb.append(",")
                if (pretty) {
                    sb.append('\n')
                    sb.append(indentLevel)
                }
            }
        }

        private fun format(number: Double): String {
            var n = number.toString()
            if (n.endsWith(".0")) {
                n = n.substring(0, n.length - 2)
            }
            return n
        }

        init {
            visited = HashSet<JsonValue>()
        }
    }
}