/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

/**
 * Default data model for single selection.
 *
 * @param <E> the type of elements that can be selected
</E> */
class SimpleSingleSelection<E> : AbstractMutableSingleSelection<E>, MutableSingleSelection<E> {
    override var selected: E?
        set(element) {
            if (isEnabled) {
                if (field !== element) {
                    if (field == null || field != element) {
                        val previousSelected = field
                        val previousActive = field != null
                        field = element
                        notifySingleSelectedChanged(
                            SingleSelectionEvent<E>(
                                this,
                                previousActive,
                                element != null,
                                previousSelected,
                                element
                            )
                        )
                    }
                }
            }
        }

    constructor() : super() {
        this.selected = null
    }

    constructor(selected: E) : super() {
        this.selected = selected
    }

    override fun clearSelection(): Boolean {
        return if (isEnabled && isActive) {
            selected = null
            true
        } else {
            false
        }
    }

    override val isActive: Boolean
        get() = selected != null && isEnabled

    override fun isSelected(element: E?): Boolean {
        return selected != null && selected == element
    }

    override fun setSelectedState(element: E, selected: Boolean) {
        if (isEnabled) {
            if (selected) {
                this.selected = element
            } else {
                if (isSelected(element)) {
                    clearSelection()
                }
            }
        }
    }

    override fun setSelectedIterableState(element: Iterable<E>, selected: Boolean): Boolean {
        throw UnsupportedOperationException()
    }

    fun setSelected(element: MutableSet<E?>?, selected: Boolean): Boolean {
        throw UnsupportedOperationException()
    }

    override fun setSelectedElementsState(selected: Boolean, vararg elements: E): Boolean {
        throw UnsupportedOperationException()
    }

    override fun setSelectedElements(vararg elements: E) {
        throw UnsupportedOperationException()
    }

    override fun setSelectedIterable(elements: Iterable<E>) {
        throw UnsupportedOperationException()
    }

    override fun setSelectedElement(element: E) {
        selected = element
    }

    override val selectedCount: Int
        get() = if (isActive) 1 else 0

    override val selectedSet: Set<E>
        get() = if (selected != null) setOf(selected!!) else emptySet()

    override operator fun iterator(): Iterator<E> {
        return object : Iterator<E> {
            var fetched = !isActive
            override fun hasNext(): Boolean {
                return !fetched
            }

            override fun next(): E {
                fetched = true
                val s = selected
                if (s != null) {
                    return s
                } else {
                    throw IllegalStateException()
                }
            }
        }
    }

    override fun toString(): String {
        return "SimpleSingleSelection{" +
                "selected=" + selected +
                '}'
    }
}