package com.macrofocus.common.properties

import com.macrofocus.common.properties.PropertyDelegates.immutable
import com.macrofocus.common.properties.PropertyDelegates.mutable
import kotlin.test.Test
import kotlin.test.assertEquals

class PropertiesTest {
    enum class Types {
        Foo,
        Bar
    }

    @Test
    fun simpleInt() {
        val properties = SimpleProperties()
        properties.addProperty("foo", SimpleProperty(10))
        properties.addProperty("bar", SimpleProperty(20))

        assertEquals(properties.getValue("foo"), 10)
        assertEquals(properties.getValue("bar"), 20)

        properties.replaceProperty("bar", properties.getProperty("foo"))

        assertEquals(properties.getValue("foo"), 10)
        assertEquals(properties.getValue("bar"), 10)
    }

    @Test
    fun delegate() {
        val properties = SimpleProperties()
        val fooProperty = properties.addProperty("foo", SimpleProperty(10))
        val barProperty = properties.addProperty("bar", SimpleProperty(20))

        var foo by mutable(properties.getProperty("foo"))
        var bar by mutable(properties.getProperty("bar"))

        assertEquals(foo, 10)
        assertEquals(bar, 20)

        bar = 30

        assertEquals(foo, 10)
        assertEquals(bar, 30)
        assertEquals(fooProperty.value, 10)
        assertEquals(barProperty.value, 30)
    }

    @Test
    fun enumInt() {
        val properties = EnumProperties(Types.values())
        properties.addProperty(Types.Foo, SimpleProperty(10))
        properties.addProperty(Types.Bar, SimpleProperty(20))

        assertEquals(properties.getValue(Types.Foo), 10)
        assertEquals(properties.getValue(Types.Bar), 20)

        properties.replaceProperty(Types.Bar, properties.getProperty(Types.Foo))

        assertEquals(properties.getValue(Types.Foo), 10)
        assertEquals(properties.getValue(Types.Bar), 10)
    }

    @Test
    fun enumPairsInt() {
        val properties = EnumProperties(Types.values(),
            Types.Foo to 10,
            Types.Bar to 20
        )
        val foo: Int by immutable(properties.getProperty(Types.Foo) as Property<Int>)
        var bar: Int by mutable(properties.getProperty(Types.Bar) as MutableProperty<Int>)

        assertEquals(foo, 10)
        assertEquals(bar, 20)

        properties.replaceProperty(Types.Bar, properties.getProperty(Types.Foo))

        assertEquals(properties.getValue(Types.Foo), 10)
        assertEquals(properties.getValue(Types.Bar), 10)

        bar = 30

        assertEquals(properties.getValue(Types.Foo), 30)
        assertEquals(properties.getValue(Types.Bar), 30)

        assertEquals(foo, 30)
        assertEquals(bar, 30)
    }

    @Test
    fun enumCreate() {
        val properties = EnumProperties(Types.values())
        with(properties) {
            val fooProperty = createProperty(Types.Foo, 10)
            val barProperty = createProperty(Types.Bar, 20)
            val foo by immutable(fooProperty)
            var bar by mutable(barProperty)

            assertEquals(fooProperty.value, 10)
            assertEquals(barProperty.value, 20)

            replaceProperty(Types.Bar, properties.getProperty(Types.Foo))

            assertEquals(foo, 10)
            assertEquals(bar, 10)
        }
    }
}