package com.macrofocus.common.format

import com.macrofocus.common.date.CPTimeZone
import com.macrofocus.common.locale.CPLocale
import java.text.*
import java.util.*

/**
 * Created by luc on 14/12/15.
 */
actual open class FormatFactory : FormatFactoryCommon {
    fun createCurrencyFormat(fractionDigits: Int, currency: Currency?, locale: Locale?): CPFormat<Any?>? {
        val instance: NumberFormat
        instance = if (currency != null) {
            DecimalFormat.getCurrencyInstance(locale)
        } else {
            DecimalFormat.getNumberInstance(locale)
        }
        instance.maximumFractionDigits = fractionDigits
        if (currency != null) {
            instance.currency = currency
        }
        return SwingFormat(instance, null, null, 1.0, 1.0)
    }

    actual fun createDecimalFormat(): CPFormat<Any?> {
        return SwingFormat(DecimalFormat.getInstance(), null, null, 1.0, 1.0)
    }

    override actual fun createDecimalFormat(fractionDigits: Int, pre: String?, post: String?, rounding: Double, factor: Double): CPFormat<Any?> {
        val instance = DecimalFormat.getInstance()
        instance.maximumFractionDigits = fractionDigits
        return SwingFormat(instance, pre, post, rounding, factor)
    }

    override actual fun createIntegerFormat(pre: String?, post: String?, rounding: Double): CPFormat<Any?> {
        return SwingFormat(DecimalFormat.getIntegerInstance(), pre, post, rounding, 1.0)
    }

    override actual fun createPercentDecimalFormat(fractionDigits: Int, rounding: Double): CPFormat<Any?> {
        val instance = DecimalFormat.getPercentInstance()
        instance.maximumFractionDigits = fractionDigits
        return SwingFormat(instance, null, null, rounding, 1.0)
    }

    actual fun createDateFormat(): CPFormat<Any?> {
        return SwingDateFormat("dd.MM.yyyy")
    }

    actual fun createDateTimeFormat(): CPFormat<Any?> {
        return SwingDateFormat("dd.MM.yyyy HH:mm:ss")
    }

    actual fun createDateOrDateTimeFormat(): CPFormat<Any?> {
        return object : CPFormat<Any?> {
            var dateFormat: DateFormat = SimpleDateFormat("dd.MM.yyyy")
            var fullDateFormat: DateFormat = SimpleDateFormat("dd.MM.yyyy HH:mm:ss")
            override fun format(value: Any?): String? {
                return formatHtml(value, false)
            }

            override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
                return if (value != null) {
                    val date: Date
                    date = if (value is Number) {
                        Date(value.toLong())
                    } else if (value is Date) {
                        value
                    } else {
                        return value.toString()
                    }
                    if (date.hours != 0 || date.minutes != 0 || date.seconds != 0) {
                        fullDateFormat.format(date)
                    } else {
                        dateFormat.format(date)
                    }
                } else {
                    ""
                }
            }

            @Throws(CPFormat.ParsingException::class)
            override fun parse(text: String?): Any? {
                return try {
                    fullDateFormat.parse(text)
                } catch (e: ParseException) {
                    try {
                        dateFormat.parse(text)
                    } catch (e1: ParseException) {
                        throw CPFormat.ParsingException(e.message, e.errorOffset)
                    }
                }
            }

            override val pattern: String?
                get() = null
            override val nativeFormat: Any?
                get() = null
            override val horizontalAlignment: CPFormat.HorizontalAlignment?
                get() = null

            override fun getClassName(value: Any?): String? {
                return null
            }
        }
    }

    fun createFormat(format: Format): CPFormat<Any?> {
        return SwingFormat(format, null, null, 1.0, 1.0)
    }

    actual fun createNumberFormat(pattern: String?): CPFormat<Any?> {
        return SwingNumberFormat(pattern)
    }

    actual fun createDateFormat(pattern: String?): CPFormat<Any?> {
        return SwingDateFormat(pattern)
    }

    actual fun createDateFormat(pattern: String, locale: CPLocale): CPFormat<Any?> {
        return SwingDateFormat(pattern, locale)
    }

    actual fun createDateFormat(pattern: String, locale: CPLocale, zone: CPTimeZone): CPFormat<Any?> {
        return SwingDateFormat(pattern, locale, zone)
    }

    private class SwingFormat(
        override val nativeFormat: Format,
        private val pre: String?,
        private val post: String?,
        private val rounding: Double,
        private val factor: Double
    ) :
        AbstractFormat<Any?>() {
        override fun formatHtml(value: Any?, htmlSupported: Boolean): String? {
            var s: String?
            if (value is Number) {
                var n = value as Number
                if (rounding != 1.0) {
                    n = Math.round(n.toDouble() / rounding) * rounding
                }
                if (factor != 1.0) {
                    n = n.toDouble() * factor
                }
                s = nativeFormat.format(n)
                if (pre != null) {
                    s = pre + s
                }
                if (post != null) {
                    s = s + post
                }
            } else {
                s = null
            }
            return s
        }

        @Throws(CPFormat.ParsingException::class)
        override fun parse(text: String?): Any? {
            var text = text
            return if (text != null) {
                if (pre != null && !text.startsWith(pre)) {
                    text = pre + text
                }
                if (post != null && !text.endsWith(post)) {
                    text = text + post
                }
                try {
                    nativeFormat.parseObject(text)
                } catch (e: ParseException) {
                    throw CPFormat.ParsingException(e.message, e.errorOffset)
                }
            } else {
                null
            }
        }

    }

    private class SwingNumberFormat(pattern: String?) : CPFormat<Any?> {
        val numberFormat: DecimalFormat
        override val pattern: String?

        override fun format(value: Any?): String {
            return formatHtml(value, false)
        }

        override fun formatHtml(value: Any?, htmlSupported: Boolean): String {
            return if (value != null) {
                val number: Number
                number = if (value is Number) {
                    value
                } else {
                    return value.toString()
                }
                numberFormat.format(number)
            } else {
                ""
            }
        }

        @Throws(CPFormat.ParsingException::class)
        override fun parse(text: String?): Any {
            return try {
                numberFormat.parse(text)
            } catch (e: ParseException) {
                throw CPFormat.ParsingException(e.message, e.errorOffset)
            }
        }

        override val nativeFormat: Any
            get() = numberFormat
        override val horizontalAlignment: CPFormat.HorizontalAlignment?
            get() = null

        override fun getClassName(value: Any?): String? {
            return null
        }

        init {
            numberFormat = DecimalFormat(pattern)
            this.pattern = pattern
        }
    }

    private class SwingDateFormat : CPFormat<Any?> {
        val dateFormat: DateFormat
        override val pattern: String?

        constructor(pattern: String?) {
            dateFormat = SimpleDateFormat(pattern)
            this.pattern = pattern
        }

        constructor(pattern: String, locale: CPLocale) {
            dateFormat = SimpleDateFormat(pattern, locale.locale)
            this.pattern = pattern
        }

        constructor(pattern: String?, locale: CPLocale, timeZone: CPTimeZone) {
            dateFormat = SimpleDateFormat(pattern, locale.locale)
            dateFormat.timeZone = timeZone.timeZone
            this.pattern = pattern
        }

        override fun format(value: Any?): String {
            return formatHtml(value, false)
        }

        override fun formatHtml(value: Any?, htmlSupported: Boolean): String {
            return if (value != null) {
                val date: Date
                date = if (value is Number) {
                    Date(value.toLong())
                } else if (value is Date) {
                    value
                } else {
                    return value.toString()
                }
                dateFormat.format(date)
            } else {
                ""
            }
        }

        @Throws(CPFormat.ParsingException::class)
        override fun parse(text: String?): Any {
            return try {
                dateFormat.parse(text)
            } catch (e: ParseException) {
                throw CPFormat.ParsingException(e.message, e.errorOffset)
            }
        }

        override val nativeFormat: Any
            get() = dateFormat
        override val horizontalAlignment: CPFormat.HorizontalAlignment?
            get() = null

        override fun getClassName(value: Any?): String? {
            return null
        }
    }

    actual companion object {
        actual val instance: FormatFactory = FormatFactory()
    }
}