/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

actual interface JsonString : JsonValue {
    actual val string: String
}
/**
 * Server-side implementation of JsonString.
 */
class JvmJsonString(override var string: String) : JsonString {
    override fun asBoolean(): Boolean {
        return !string.isEmpty()
    }

    override fun asNumber(): Double {
        return try {
            if (asString()!!.isEmpty()) {
                0.0
            } else {
                asString()!!.toDouble()
            }
        } catch (nfe: NumberFormatException) {
            Double.NaN
        }
    }

    override fun asString(): String? {
        return string
    }

    override fun asJsonArray(): JsonArray? {
        return null
    }

    override fun asJsonObject(): JsonObject? {
        return null
    }

    override val `object`: Any
        get() = string
    override val type: JsonType
        get() = JsonType.STRING

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JsonValue).`object`
    }

    override fun toNative(): Any? {
        return this
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        visitor.visit(string, ctx)
    }

    override fun toJson(): String {
        return JsonUtil.quote(string)
    }
}