/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.filter

/**
 * This interface represents the current state of a filter. A filter is to be understood in the broad sense and
 * not only related to user interactions.
 *
 * @param <E> the type of elements that can be filtered
</E> */
interface Filter<E> {
    /**
     * Indicates whether at least one element is currently filtered.
     *
     * @return true if at least one element is filtered, false otherwise.
     */
    val isActive: Boolean

    /**
     * Indicates whether a given element is filtered or not.
     *
     * @param element the element to be tested
     * @return true if the element is filtered, false otherwise
     */
    fun isFiltered(element: E): Boolean

    /**
     * Indicates whether a given element is filtered or not by the given locksmith.
     *
     * @param element the element to be tested
     * @param locksmith    the object used to filter the element
     * @return true if the element is filtered, false otherwise
     */
    fun isFilteredBy(element: E, locksmith: Any): Boolean

    /**
     * Returns the number of elements currently filtered.
     *
     * @return the number of filtered elements.
     */
    val filteredCount: Int

    /**
     * Add a listener to the list that's notified each time a change to the filter occurs.
     *
     * @param listener the SelectionListener
     */
    fun addFilterListener(listener: FilterListener<E>)

    /**
     * Add a listener to the list that's notified each time a change to the filter occurs. The listener will
     * automatically be disposed of should no other object have a reference to it.
     *
     * @param listener the SelectionListener
     */
    fun addWeakFilterListener(listener: FilterListener<E>)

    /**
     * Remove a listener to the list that's notified each time a change to the filter occurs.
     *
     * @param listener the SelectionListener
     */
    fun removeFilterListener(listener: FilterListener<E>)

    /**
     * Remove all listeners to the list that's notified each time a change to the selection occurs.
     */
    fun removeFilterListeners()

    val isEnabled: Boolean
}