/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

/**
 * This class provides a skeletal implementation of the MutableSelection interface to minimize the effort required to implement
 * this interface.
 */
abstract class AbstractMutableSingleSelection<E> : AbstractMutableSelection<E>(), MutableSingleSelection<E> {
    private val listeners: MutableList<SingleSelectionListener<E>>
    override fun addSingleSelectionListener(listener: SingleSelectionListener<E>) {
        listeners.add(listener)
    }

    override fun addWeakSingleSelectionListener(listener: SingleSelectionListener<E>) {
        val weakListener = WeakSingleSelectionListener(listener)
        listeners.add(weakListener)
    }

    override fun removeSingleSelectionListener(listener: SingleSelectionListener<E>) {
        if (listener is WeakSingleSelectionListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: SingleSelectionListener<E>? = null
            for (selectionListener in listeners) {
                var comparable: SingleSelectionListener<E>?
                comparable = if (selectionListener is WeakSingleSelectionListener) {
                    selectionListener.getReference()
                } else {
                    selectionListener
                }
                if (listener.equals(comparable)) {
                    toRemove = selectionListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    protected fun notifySingleSelectedChanged(event: SingleSelectionEvent<E>) {
        for (listener in listeners) {
            listener.selectionChanged(event)
        }
        super.notifySelectedChanged(event)
    }

    private inner class WeakSingleSelectionListener(listener: SingleSelectionListener<E>) : SingleSelectionListener<E> {
        private val l_ref: WeakReference<SingleSelectionListener<E>>
        override fun selectionChanged(event: SingleSelectionEvent<E>) {
            val l: SingleSelectionListener<E>? = getReference()
            if (l != null) {
                l.selectionChanged(event)
            } else {
                removeSingleSelectionListener(this)
            }
        }

        fun getReference(): SingleSelectionListener<E>? {
            return l_ref.get()
        }

        override fun toString(): String {
            val l: SingleSelectionListener<E>? = getReference()
            return if (l != null) {
                "Weak[$l]"
            } else {
                this::class.toString() + "@" + hashCode()
            }
        }

        init {
            l_ref = WeakReference<SingleSelectionListener<E>>(listener)
        }
    }

    init {
        listeners = CollectionFactory.copyOnWriteArrayList<SingleSelectionListener<E>>()
    }
}