/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

actual interface JsonArray : JsonValue {
    /**
     * Return the ith element of the array.
     */
    operator fun <T : JsonValue?> get(index: Int): T

    /**
     * Return the ith element of the array (uncoerced) as a JsonArray. If the type is not an array,
     * this can result in runtime errors.
     */
    fun getArray(index: Int): JsonArray?

    /**
     * Return the ith element of the array (uncoerced) as a boolean. If the type is not a boolean,
     * this can result in runtime errors.
     */
    fun getBoolean(index: Int): Boolean

    /**
     * Return the ith element of the array (uncoerced) as a number. If the type is not a number, this
     * can result in runtime errors.
     */
    fun getNumber(index: Int): Double

    /**
     * Return the ith element of the array (uncoerced) as a JsonObject If the type is not an object,,
     * this can result in runtime errors.
     */
    fun getObject(index: Int): JsonObject?

    /**
     * Return the ith element of the array (uncoerced) as a String. If the type is not a String, this
     * can result in runtime errors.
     */
    fun getString(index: Int): String?

    /**
     * Length of the array.
     */
    fun length(): Int

    /**
     * Remove an element of the array at a particular index.
     */
    fun remove(index: Int)

    /**
     * Set the value at index to be a given value.
     */
    operator fun set(index: Int, value: JsonValue?)

    /**
     * Set the value at index to be a String value.
     */
    operator fun set(index: Int, string: String)

    /**
     * Set the value at index to be a number value.
     */
    operator fun set(index: Int, number: Double)

    /**
     * Set the value at index to be a boolean value.
     */
    operator fun set(index: Int, bool: Boolean)
}

/**
 * Server-side implementation of JsonArray.
 */
class JvmJsonArray(factory: JsonFactory) : JsonArray {
    private var arrayValues: ArrayList<JsonValue> = ArrayList<JsonValue>()

    private var factory: JsonFactory
    override fun asBoolean(): Boolean {
        return true
    }

    override fun asNumber(): Double {
        return when (length()) {
            0 -> 0.0
            1 -> (get(0) as JsonValue).asNumber()
            else -> Double.NaN
        }
    }

    override fun asString(): String {
        val toReturn = StringBuilder()
        for (i in 0 until length()) {
            if (i > 0) {
                toReturn.append(", ")
            }
            toReturn.append((get(i) as JsonValue).asString())
        }
        return toReturn.toString()
    }

    override fun asJsonArray(): JsonArray? {
        return this
    }

    override fun asJsonObject(): JsonObject? {
        return null
    }

    override operator fun <T : JsonValue?> get(index: Int): T {
        return arrayValues.get(index) as T
    }

    override fun getArray(index: Int): JsonArray? {
        return get(index) as JsonArray?
    }

    override fun getBoolean(index: Int): Boolean {
        return (get(index) as JsonBoolean).boolean
    }

    override fun getNumber(index: Int): Double {
        return (get(index) as JsonNumber).number
    }

    override fun getObject(index: Int): JsonObject? {
        return get(index) as JsonObject?
    }

    override val `object`: Any
        get() {
            val objs: MutableList<Any> = ArrayList<Any>()
            for (`val` in arrayValues) {
                objs.add((`val` as JsonValue).`object`)
            }
            return objs
        }

    override fun getString(index: Int): String? {
        return (get(index) as JsonString).string
    }

    override val type: JsonType
        get() = JsonType.ARRAY

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JsonValue).`object`
    }

    override fun length(): Int {
        return arrayValues.size
    }

    override fun remove(index: Int) {
        arrayValues.removeAt(index)
    }

    override operator fun set(index: Int, value: JsonValue?) {
        var value: JsonValue? = value
        if (value == null) {
            value = factory.createNull()
        }
        if (index == arrayValues.size) {
            arrayValues.add(index, value)
        } else {
            arrayValues.set(index, value)
        }
    }

    override operator fun set(index: Int, string: String) {
        set(index, factory.create(string))
    }

    override operator fun set(index: Int, number: Double) {
        set(index, factory.create(number))
    }

    override operator fun set(index: Int, bool: Boolean) {
        set(index, factory.create(bool))
    }

    override fun toNative(): Any? {
        return this
    }

    override fun toJson(): String {
        return JsonUtil.stringify(this)
    }

    override fun traverse(
        visitor: JsonVisitor,
        ctx: JsonContext
    ) {
        if (visitor.visit(this, ctx)) {
            val arrayCtx = JsonArrayContext(this)
            for (i in 0 until length()) {
                arrayCtx.currentIndex = i
                if (visitor.visitIndex(arrayCtx.currentIndex, arrayCtx)) {
                    visitor.accept(get(i), arrayCtx)
                    arrayCtx.isFirst = false
                }
            }
        }
        visitor.endVisit(this, ctx)
    }

    init {
        this.factory = factory
    }
}

/**
 * Return the ith element of the array.
 */
actual operator fun <T : JsonValue?> JsonArray.get(index: Int): T {
    return this.get(index)
}

/**
 * Return the ith element of the array (uncoerced) as a JsonArray. If the type is not an array,
 * this can result in runtime errors.
 */
actual fun JsonArray.getArray(index: Int): JsonArray? {
    return this.getArray(index)
}

/**
 * Return the ith element of the array (uncoerced) as a boolean. If the type is not a boolean,
 * this can result in runtime errors.
 */
actual fun JsonArray.getBoolean(index: Int): Boolean {
    return this.getBoolean(index)
}

/**
 * Return the ith element of the array (uncoerced) as a number. If the type is not a number, this
 * can result in runtime errors.
 */
actual fun JsonArray.getNumber(index: Int): Double {
    return this.getNumber(index)
}

/**
 * Return the ith element of the array (uncoerced) as a JsonObject If the type is not an object,,
 * this can result in runtime errors.
 */
actual fun JsonArray.getObject(index: Int): JsonObject? {
    return this.getObject(index)
}

/**
 * Return the ith element of the array (uncoerced) as a String. If the type is not a String, this
 * can result in runtime errors.
 */
actual fun JsonArray.getString(index: Int): String? {
    return this.getString(index)
}

/**
 * Length of the array.
 */
actual fun JsonArray.length(): Int {
    return this.length()
}

/**
 * Remove an element of the array at a particular index.
 */
actual fun JsonArray.remove(index: Int) {
    this.remove(index)
}

/**
 * Set the value at index to be a given value.
 */
actual operator fun JsonArray.set(index: Int, value: JsonValue?) {
    this.set(index, value)
}

/**
 * Set the value at index to be a String value.
 */
actual operator fun JsonArray.set(index: Int, string: String) {
    this.set(index, string)
}

/**
 * Set the value at index to be a number value.
 */
actual operator fun JsonArray.set(index: Int, number: Double) {
    this.set(index, number)
}

/**
 * Set the value at index to be a boolean value.
 */
actual operator fun JsonArray.set(index: Int, bool: Boolean) {
    this.set(index, bool)
}

