/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

actual interface JsonObject : JsonValue {
    /**
     * Return the element (uncoerced) as a JsonValue.
     */
    operator fun <T : JsonValue?> get(key: String): T

    /**
     * Return the element (uncoerced) as a JsonArray. If the type is not an array,
     * this can result in runtime errors.
     */
    fun getArray(key: String): JsonArray?

    /**
     * Return the element (uncoerced) as a boolean. If the type is not a boolean,
     * this can result in runtime errors.
     */
    fun getBoolean(key: String): Boolean

    /**
     * Return the element (uncoerced) as a number. If the type is not a number, this
     * can result in runtime errors.
     */
    fun getNumber(key: String): Double

    /**
     * Return the element (uncoerced) as a JsonObject If the type is not an object,,
     * this can result in runtime errors.
     */
    fun getObject(key: String): JsonObject?

    /**
     * Return the element (uncoerced) as a String. If the type is not a String, this
     * can result in runtime errors.
     */
    fun getString(key: String): String?

    /**
     * All keys of the object.
     */
    fun keys(): Array<String>

    /**
     * Set a given key to the given value.
     */
    fun put(key: String, value: JsonValue?)

    /**
     * Set a given key to the given String value.
     */
    fun put(key: String, value: String)

    /**
     * Set a given key to the given double value.
     */
    fun put(key: String, value: Double)

    /**
     * Set a given key to the given boolean value.
     */
    fun put(key: String, bool: Boolean)

    /**
     * Test whether a given key has present.
     */
    fun hasKey(key: String): Boolean

    /**
     * Remove a given key and associated value from the object.
     * @param key
     */
    fun remove(key: String?)
}

/**
 * Return the element (uncoerced) as a JsonValue.
 */
actual operator fun <T : JsonValue?> JsonObject.get(key: String): T {
    return this.get(key)
}


/**
 * Return the element (uncoerced) as a JsonArray. If the type is not an array,
 * this can result in runtime errors.
 */
actual fun JsonObject.getArray(key: String): JsonArray? {
    return this.getArray(key)
}

/**
 * Return the element (uncoerced) as a boolean. If the type is not a boolean,
 * this can result in runtime errors.
 */
actual fun JsonObject.getBoolean(key: String): Boolean {
    return this.getBoolean(key)
}

/**
 * Return the element (uncoerced) as a number. If the type is not a number, this
 * can result in runtime errors.
 */
actual fun JsonObject.getNumber(key: String): Double {
    return this.getNumber(key)
}

/**
 * Return the element (uncoerced) as a JsonObject If the type is not an object,,
 * this can result in runtime errors.
 */
actual fun JsonObject.getObject(key: String): JsonObject? {
    return this.getObject(key)
}

/**
 * Return the element (uncoerced) as a String. If the type is not a String, this
 * can result in runtime errors.
 */
actual fun JsonObject.getString(key: String): String? {
    return this.getString(key)
}

/**
 * All keys of the object.
 */
actual fun JsonObject.keys(): Array<String> {
    return this.keys()
}

/**
 * Set a given key to the given value.
 */
actual fun JsonObject.put(key: String, value: JsonValue?) {
    this.put(key, value)
}

/**
 * Set a given key to the given String value.
 */
actual fun JsonObject.put(key: String, value: String) {
    this.put(key, value)
}

/**
 * Set a given key to the given double value.
 */
actual fun JsonObject.put(key: String, value: Double) {
    this.put(key, value)
}

/**
 * Set a given key to the given boolean value.
 */
actual fun JsonObject.put(key: String, bool: Boolean) {
    this.put(key, bool)
}

/**
 * Test whether a given key has present.
 */
actual fun JsonObject.hasKey(key: String): Boolean {
    return this.hasKey(key)
}

/**
 * Remove a given key and associated value from the object.
 * @param key
 */
actual fun JsonObject.remove(key: String?) {
    return this.remove(key)
}

/**
 * Server-side implementation of JsonObject.
 */
class JvmJsonObject(factory: JsonFactory) : JsonObject {
    private var factory: JsonFactory

    private var map: MutableMap<String, JsonValue?> = LinkedHashMap<String, JsonValue?>()
    override fun asBoolean(): Boolean {
        return true
    }

    override fun asNumber(): Double {
        return Double.NaN
    }

    override fun asString(): String {
        return "[object Object]"
    }

    override fun asJsonArray(): JsonArray? {
        return null
    }

    override fun asJsonObject(): JsonObject? {
        return this
    }

    override operator fun <T : JsonValue?> get(key: String): T {
        return map[key] as T
    }

    override fun getArray(key: String): JsonArray? {
        return get(key) as JsonArray?
    }

    override fun getBoolean(key: String): Boolean {
        return (get(key) as JsonBoolean).boolean
    }

    override fun getNumber(key: String): Double {
        return (get(key) as JsonNumber).number
    }

    override fun getObject(key: String): JsonObject? {
        return get(key) as JsonObject?
    }

    override fun toNative(): Any? {
        return this
    }

    override val `object`: Any
        get() {
            val obj: MutableMap<String, Any> = HashMap()
            for ((key, value) in map) {
                obj[key] = (value as JsonValue).`object`
            }
            return obj
        }

    override fun getString(key: String): String? {
        return (get(key) as JsonString).string
    }

    override val type: JsonType
        get() = JsonType.OBJECT

    override fun hasKey(key: String): Boolean {
        return map.containsKey(key)
    }

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JsonValue).`object`
    }

    override fun keys(): Array<String> {
        return map.keys.toTypedArray()
    }

    override fun put(key: String, value: JsonValue?) {
        var value: JsonValue? = value
        if (value == null) {
            value = factory.createNull()
        }
        map[key] = value
    }

    override fun put(key: String, value: String) {
        put(key, factory.create(value))
    }

    override fun put(key: String, value: Double) {
        put(key, factory.create(value))
    }

    override fun put(key: String, bool: Boolean) {
        put(key, factory.create(bool))
    }

    override fun remove(key: String?) {
        map.remove(key)
    }

    operator fun set(key: String, value: JsonValue?) {
        put(key, value)
    }

    override fun toJson(): String {
        return JsonUtil.stringify(this)
    }

    override fun toString(): String {
        return toJson()
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        if (visitor.visit(this, ctx)) {
            val objCtx = JsonObjectContext(this)
            for (key in stringifyOrder(keys())) {
                objCtx.currentKey = key
                if (visitor.visitKey(objCtx.currentKey, objCtx)) {
                    visitor.accept(get(key), objCtx)
                    objCtx.isFirst = false
                }
            }
        }
        visitor.endVisit(this, ctx)
    }


    companion object {
        private fun stringifyOrder(keys: Array<String>): List<String> {
            val toReturn: MutableList<String> = ArrayList<String>()
            val nonNumeric: MutableList<String> = ArrayList<String>()
            for (key in keys) {
                if (key.matches("\\d+".toRegex())) {
                    toReturn.add(key)
                } else {
                    nonNumeric.add(key)
                }
            }
            toReturn.sort()
            toReturn.addAll(nonNumeric)
            return toReturn
        }
    }

    init {
        this.factory = factory
    }
}