/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.filter

/**
 * A simple mutable filter implementation.
 */
class SimpleFilter<E>() : AbstractMutableFilter<E>() {
    private val filtered: MutableMap<E, MutableSet<Any>>

    override val filteredCount: Int
        get() = filtered.size

    override val isActive: Boolean
        get() = !filtered.isEmpty()

    override fun isFiltered(element: E): Boolean {
        return filtered.containsKey(element)
    }

    override fun clearFilter() {
        if (isEnabled && isActive) {
            val changes: MutableSet<E> = HashSet()
            changes.addAll(filtered.keys)
            filtered.clear()
            notifyFilteredChanged(SimpleFilterEvent<E>(this, emptySet(), changes))
        }
    }

    override fun clearFilterBy(locksmith: Any) {
        if (isEnabled && isActive) {
            val toRemove: MutableList<E> = ArrayList<E>()
            for (element in filtered.keys) {
                val set: MutableSet<*> = filtered[element]!!
                if (set.contains(locksmith)) {
                    set.remove(locksmith)
                    if (set.isEmpty()) {
                        toRemove.add(element)
                    }
                }
            }
            if (!toRemove.isEmpty()) {
                val changes: MutableSet<E> = HashSet()
                for (e in toRemove) {
                    changes.add(e)
                    filtered.remove(e)
                }
                notifyFilteredChanged(SimpleFilterEvent<E>(this, emptySet(), changes))
            }
        }
    }

    override fun isFilteredBy(element: E, locksmith: Any): Boolean {
        return if (filtered.containsKey(element)) {
            val set: Set<Any>? = filtered[element]
            set != null && set.contains(locksmith)
        } else {
            false
        }
    }

    override fun setFiltered(element: E, filtered: Boolean, locksmith: Any) {
        if (isEnabled) {
            if (filtered) {
                if (!this.filtered.containsKey(element)) {
                    val set: MutableSet<Any> = HashSet()
                    set.add(locksmith)
                    this.filtered[element] = set
                    val changes: MutableSet<E> = HashSet()
                    changes.add(element)
                    notifyFilteredChanged(SimpleFilterEvent<E>(this, changes, emptySet()))
                } else {
                    val set = this.filtered[element]!!
                    set.add(locksmith)
                }
            } else {
                if (this.filtered.containsKey(element)) {
                    val set: MutableSet<*> = this.filtered[element]!!
                    set.remove(locksmith)
                    if (set.isEmpty()) {
                        this.filtered.remove(element)
                        val changes: MutableSet<E> = HashSet()
                        changes.add(element)
                        notifyFilteredChanged(SimpleFilterEvent<E>(this, emptySet(), changes))
                    }
                }
            }
        }
    }

    override fun setFilteredIterable(elements: Iterable<E>, filtered: Boolean, locksmith: Any) {
        if (isEnabled) {
            val changes: MutableSet<E> = HashSet()
            for (element in elements) {
                if (filtered) {
                    if (!this.filtered.containsKey(element)) {
                        val set: MutableSet<Any> = HashSet()
                        set.add(locksmith)
                        this.filtered[element] = set
                        changes.add(element)
                    } else {
                        val set = this.filtered[element]!!
                        set.add(locksmith)
                    }
                } else {
                    if (this.filtered.containsKey(element)) {
                        val set: MutableSet<*> = this.filtered[element]!!
                        set.remove(locksmith)
                        if (set.isEmpty()) {
                            this.filtered.remove(element)
                            changes.add(element)
                        }
                    }
                }
            }
            if (changes.size > 0) {
                if (filtered) {
                    notifyFilteredChanged(SimpleFilterEvent<E>(this, changes, emptySet()))
                } else {
                    notifyFilteredChanged(SimpleFilterEvent<E>(this, emptySet(), changes))
                }
            }
        }
    }

    override fun setFilteredState(filtered: Iterable<E>, unfiltered: Iterable<E>, locksmith: Any) {
        if (isEnabled) {
            val f: MutableSet<E> = HashSet()
            val u: MutableSet<E> = HashSet()
            for (element in filtered) {
                if (!this.filtered.containsKey(element)) {
                    val set: MutableSet<Any> = HashSet()
                    set.add(locksmith)
                    this.filtered[element] = set
                    f.add(element)
                } else {
                    val set = this.filtered[element]!!
                    set.add(locksmith)
                }
            }
            for (element in unfiltered) {
                if (this.filtered.containsKey(element)) {
                    val set: MutableSet<*> = this.filtered[element]!!
                    set.remove(locksmith)
                    if (set.isEmpty()) {
                        this.filtered.remove(element)
                        u.add(element)
                    }
                }
            }
            if (f.size > 0 || u.size > 0) {
                notifyFilteredChanged(SimpleFilterEvent<E>(this, f, u))
            }
        }
    }

    operator fun iterator(): Iterator<E> {
        return filtered.keys.iterator()
    }

    companion object {
        private const val serialVersionUID = -8732563010323502592L
    }

    init {
        filtered = HashMap()
    }
}