package com.macrofocus.common.interval

import com.macrofocus.common.collection.CollectionFactory.copyOnWriteArrayList
import com.macrofocus.common.collection.WeakReference
import kotlin.math.abs

abstract class AbstractBoundedInterval protected constructor() : AbstractMutableInterval(), BoundedInterval {
    private val listeners: MutableList<BoundedIntervalListener>
    override val isFullRange: Boolean
        get() = extent == abs(maximum - minimum)

    override fun addBoundedIntervalListener(listener: BoundedIntervalListener) {
        listeners.add(listener)
    }

    override fun addWeakBoundedIntervalListener(listener: BoundedIntervalListener) {
        val weakListener: WeakBoundedIntervalListener = WeakBoundedIntervalListener(listener)
        listeners.add(weakListener)
    }

    override fun removeBoundedIntervalListener(listener: BoundedIntervalListener) {
        if (listener is WeakBoundedIntervalListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: BoundedIntervalListener? = null
            for (intervalListener in listeners) {
                val comparable: BoundedIntervalListener?
                if (intervalListener is WeakBoundedIntervalListener) {
                    comparable = intervalListener.reference
                } else {
                    comparable = intervalListener
                }
                if (listener == comparable) {
                    toRemove = intervalListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    override fun removeBoundedIntervalListeners() {
        listeners.clear()
    }

    protected fun notifyBoundedIntervalChanged(event: BoundedIntervalEvent) {
        for (listener in listeners) {
            listener.boundedIntervalChanged(event)
        }
    }

    private inner class WeakBoundedIntervalListener(listener: BoundedIntervalListener) :
        BoundedIntervalListener {
        private val l_ref: WeakReference<BoundedIntervalListener>
        override fun boundedIntervalChanged(event: BoundedIntervalEvent) {
            val l = reference
            if (l != null) {
                l.boundedIntervalChanged(event)
            } else {
                removeBoundedIntervalListener(this)
            }
        }

        val reference: BoundedIntervalListener
            get() = l_ref.get()

        override fun toString(): String {
            val l = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                this::class.toString() + "@" + hashCode()
            }
        }

        init {
            l_ref = WeakReference<BoundedIntervalListener>(listener)
        }
    }

    init {
        listeners = copyOnWriteArrayList<BoundedIntervalListener>()
    }
}
