/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.collection

import java.util.concurrent.CopyOnWriteArrayList

class ConcurrentLinkedHashSet<K> : MutableSet<K> {
    private val set: ConcurrentHashSet<K> = ConcurrentHashSet<K>()
    private val holder: CopyOnWriteArrayList<K> = CopyOnWriteArrayList<K>()

    override fun add(element: K): Boolean {
        if (set.contains(element)) return false
        synchronized(set) {
            set.add(element)
            holder.add(element)
        }
        return true
    }

    override fun addAll(elements: Collection<K>): Boolean {
        var changed = false
        for (element in elements) {
            changed = add(element) || changed
        }
        return changed
    }

    override fun clear() {
        synchronized(set) {
            set.clear()
            holder.clear()
        }
    }

    override fun contains(element: K): Boolean {
        return set.contains(element)
    }

    override fun containsAll(elements: Collection<K>): Boolean {
        for (o in elements) {
            if (!contains(o)) return false
        }
        return true
    }

    override fun isEmpty(): Boolean {
        return holder.size == 0
    }

    override fun iterator(): MutableIterator<K> {
        return holder.iterator()
    }

    override fun remove(element: K): Boolean {
        if (!set.contains(element)) return false
        synchronized(set) {
            set.remove(element)
            holder.remove(element)
        }
        return true
    }

    override fun removeAll(elements: Collection<K>): Boolean {
        var changed = false
        for (o in elements) {
            changed = remove(o) || changed
        }
        return changed
    }

    override fun retainAll(elements: Collection<K>): Boolean {
        clear()
        for (element in elements) {
            add(element)
        }
        return true
    }

    override val size: Int
        get() =  holder.size

    fun toArray(): Array<Any?> {
        return holder.toTypedArray()
    }

    fun <T : Any?> toArray(a: Array<T>): Array<T> {
        return holder.toArray(a)
    }
}