/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream

/**
 * Server-side implementation of JsonArray.
 */
class JavaJsonArray(factory: JsonFactory) : JavaJsonValue(), JsonArray {
    private var arrayValues: ArrayList<JsonValue> = ArrayList<JsonValue>()

    private var factory: JsonFactory
    override fun asBoolean(): Boolean {
        return true
    }

    override fun asNumber(): Double {
        return when (length()) {
            0 -> 0.0
            1 -> (get(0) as JsonValue).asNumber()
            else -> Double.NaN
        }
    }

    override fun asString(): String {
        val toReturn = StringBuilder()
        for (i in 0 until length()) {
            if (i > 0) {
                toReturn.append(", ")
            }
            toReturn.append((get(i) as JsonValue).asString())
        }
        return toReturn.toString()
    }

    override operator fun <T : JsonValue?> get(index: Int): T {
        return arrayValues.get(index) as T
    }

    override fun getArray(index: Int): JsonArray {
        return get(index) as JsonArray
    }

    override fun getBoolean(index: Int): Boolean {
        return (get(index) as JsonBoolean).boolean
    }

    override fun getNumber(index: Int): Double {
        return (get(index) as JsonNumber).number
    }

    override fun getObject(index: Int): JsonObject {
        return get(index) as JsonObject
    }

    override val `object`: Any
        get() {
            val objs: MutableList<Any> = java.util.ArrayList<Any>()
            for (`val` in arrayValues) {
                objs.add((`val` as JavaJsonValue).`object`)
            }
            return objs
        }

    override fun getString(index: Int): String {
        return (get(index) as JsonString).string
    }

    override val type: JsonType
        get() = JsonType.ARRAY

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JavaJsonValue).`object`
    }

    override fun length(): Int {
        return arrayValues.size
    }

    override fun remove(index: Int) {
        arrayValues.removeAt(index)
    }

    override operator fun set(index: Int, value: JsonValue?) {
        var value: JsonValue? = value
        if (value == null) {
            value = factory.createNull()
        }
        if (index == arrayValues.size) {
            arrayValues.add(index, value)
        } else {
            arrayValues.set(index, value)
        }
    }

    override operator fun set(index: Int, string: String) {
        set(index, factory.create(string))
    }

    override operator fun set(index: Int, number: Double) {
        set(index, factory.create(number))
    }

    override operator fun set(index: Int, bool: Boolean) {
        set(index, factory.create(bool))
    }

    override fun toJson(): String {
        return JsonUtil.stringify(this)
    }

    override fun traverse(
        visitor: JsonVisitor,
        ctx: JsonContext
    ) {
        if (visitor.visit(this, ctx)) {
            val arrayCtx = JsonArrayContext(this)
            for (i in 0 until length()) {
                arrayCtx.currentIndex = i
                if (visitor.visitIndex(arrayCtx.currentIndex, arrayCtx)) {
                    visitor.accept(get(i), arrayCtx)
                    arrayCtx.isFirst = false
                }
            }
        }
        visitor.endVisit(this, ctx)
    }

    @Throws(IOException::class, ClassNotFoundException::class)
    private fun readObject(stream: ObjectInputStream) {
        val instance: JavaJsonArray = parseJson(stream)
        factory = Json.instance()
        arrayValues = instance.arrayValues
    }

    @Throws(IOException::class)
    private fun writeObject(stream: ObjectOutputStream) {
        stream.writeObject(toJson())
    }

    companion object {
        private const val serialVersionUID = 1L
    }

    init {
        this.factory = factory
    }
}