package com.macrofocus.common.interval

import com.macrofocus.common.properties.MutableProperty
import kotlin.jvm.JvmOverloads
import kotlin.math.min

/**
 * / ** Default data model for bounded interval.
 */
class SimpleBoundedInterval(private val interval: MutableInterval, minimum: Double, maximum: Double) :
    AbstractBoundedInterval(), MutableBoundedInterval {
    override var minimum = 0.0
        set(minimum) {
            if (field != minimum) {
                val event = BoundedIntervalEvent(field, maximum, minimumExtent, maximumExtent)
                field = minimum

                //        minimumExtent = 0;
                maximumExtent = maximum - field
                event.setNewValues(field, maximum, minimumExtent, maximumExtent)
                notifyBoundedIntervalChanged(event)
            }
        }
    override var maximum = 0.0
        set(maximum) {
            if (field != maximum) {
                val event = BoundedIntervalEvent(minimum, field, minimumExtent, maximumExtent)
                field = maximum

                //        minimumExtent = 0;
                maximumExtent = this.maximum - minimum
                event.setNewValues(minimum, field, minimumExtent, maximumExtent)
                notifyBoundedIntervalChanged(event)
            }

        }
    override var minimumExtent = 0.0
        set(minimumExtent) {
            if (field != minimumExtent) {
                val event = BoundedIntervalEvent(minimum, maximum, field, maximumExtent)
                field = minimumExtent
                event.setNewValues(minimum, maximum, field, maximumExtent)
                notifyBoundedIntervalChanged(event)
            }
        }

    override var maximumExtent = 0.0
        set(maximumExtent) {
            if (field != maximumExtent) {
                val event = BoundedIntervalEvent(minimum, maximum, minimumExtent, field)
                field = maximumExtent
                event.setNewValues(minimum, maximum, minimumExtent, field)
                notifyBoundedIntervalChanged(event)
            }
        }
    private val intervalListener: IntervalListener = object : IntervalListener {
        override fun intervalChanged(event: IntervalEvent?) {
            notifyIntervalChanged(event)
        }
    }

    @JvmOverloads
    constructor(start: Double, extent: Double, minimum: Double = start, maximum: Double = start + extent) : this(
        PropertyClosedInterval(start, extent),
        minimum,
        maximum
    ) {
    }

    override fun setValue(value: Double, extent: Double) {
        interval.setValue(value, extent)
    }

    override val startProperty: MutableProperty<Double>
        get() = interval.startProperty
    override val endProperty: MutableProperty<Double>
        get() = interval.endProperty

    override var start: Double
        get() = interval.start
        set(value) {
            this.interval.start = value
        }
    override var end: Double
        get() = interval.end
        set(value) {
            interval.end = value
        }
    override var extent: Double
        get() = interval.extent
        set(value) {
            interval.extent = value
        }

    override fun contains(value: Double): Boolean {
        return interval.contains(value)
    }

    override fun containsStartEnd(start: Double, end: Double): Boolean {
        return interval.containsStartEnd(start, end)
    }

    override fun containsInterval(interval: Interval?): Boolean {
        return interval!!.containsInterval(interval)
    }

    override fun overlaps(start: Double, end: Double): Boolean {
        return interval.overlaps(start, end)
    }

    override fun overlapsInterval(interval: Interval?): Boolean {
        return interval!!.overlapsInterval(interval)
    }

    override val isDegenerate: Boolean
        get() = interval.isDegenerate
    override val isInverted: Boolean
        get() = interval.isInverted
    override val isInvertedScale: Boolean
        get() = minimum > maximum


    override fun setMinMax(min: Double, max: Double) {
        if (minimum != min || maximum != max) {
            val event = BoundedIntervalEvent(minimum, maximum, minimumExtent, maximumExtent)
            minimum = min
            maximum = max

            //        minimumExtent = 0;
            maximumExtent = max - min
            event.setNewValues(minimum, maximum, minimumExtent, maximumExtent)
            notifyBoundedIntervalChanged(event)
        }
    }

    override fun setMinMaxExtent(min: Double, max: Double, minExtent: Double, maxExtent: Double) {
        if (minimum != min || maximum != max || minimumExtent != minExtent || maximumExtent != maximumExtent) {
            val event = BoundedIntervalEvent(minimum, maximum, minimumExtent, maximumExtent)
            minimum = min
            maximum = max
            minimumExtent = minExtent
            maximumExtent = maxExtent
            event.setNewValues(minimum, maximum, minimumExtent, maximumExtent)
            notifyBoundedIntervalChanged(event)
        }
    }

    override fun reset() {
        setValue(minimum, min(minimum + maximumExtent, maximum) - minimum)
    }

    override fun toString(): String {
        return "SimpleBoundedInterval{" +
                "start=" + interval.start +
                ", end=" + interval.end +
                ", extent=" + interval.extent +
                ", minimum=" + minimum +
                ", maximum=" + maximum +
                ", minimumExtent=" + minimumExtent +
                ", maximumExtent=" + maximumExtent +
                '}'
    }

    init {
        this.minimum = minimum
        this.maximum = maximum
        minimumExtent = 0.0
        maximumExtent = maximum - minimum
        interval.addWeakIntervalListener(intervalListener)
    }
}
