/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.macrofocus.common.json

import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream

/**
 * Server-side implementation of JsonNumber.
 */
class JavaJsonNumber(override var number: Double) : JavaJsonValue(), JsonNumber {
    override fun asBoolean(): Boolean {
        return if (java.lang.Double.isNaN(number) || java.lang.Math.abs(number) == 0.0) false else true
    }

    override fun asNumber(): Double {
        return number
    }

    override fun asString(): String {
        return toJson()
    }

    override val `object`: Any
        get() = number
    override val type: com.macrofocus.common.json.JsonType
        get() = JsonType.NUMBER

    override fun jsEquals(value: JsonValue?): Boolean {
        return `object` == (value as JavaJsonValue).`object`
    }

    override fun traverse(visitor: JsonVisitor, ctx: JsonContext) {
        visitor.visit(number, ctx)
    }

    override fun toJson(): String {
        if (java.lang.Double.isInfinite(number) || java.lang.Double.isNaN(number)) {
            return "null"
        }
        var toReturn = number.toString()
        if (toReturn.endsWith(".0")) {
            toReturn = toReturn.substring(0, toReturn.length - 2)
        }
        return toReturn
    }

    @Throws(IOException::class, ClassNotFoundException::class)
    private fun readObject(stream: ObjectInputStream) {
        val instance: JavaJsonNumber = parseJson(stream)
        number = instance.number
    }

    @Throws(IOException::class)
    private fun writeObject(stream: ObjectOutputStream) {
        stream.writeObject(toJson())
    }

    companion object {
        private const val serialVersionUID = 1L
    }
}